import argparse
import json
import os
import typing

import juggler_sdk

from pprint import pprint
from library.python.vault_client.instances import Production as VaultClient

from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager.config import JUGGLER_API_URL, \
    TENTACLES_VAULT_SECRET_UUID
from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager.factory import JugglerChecksFactory
from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager import golovan
from infra.rtc_sla_tentacles.backend.lib.util import read_yaml_file

BACKEND_CHECK_GROUP_NAMES = [
    "harvesters_snapshots_freshness",
    "reallocation",
    "redeployment",
    "slo",
    "common",
]


def get_juggler_oauth_token_from_vault() -> typing.AnyStr:
    """
        Returns Juggler API OAuth token stored in Ya Vault.
    """
    client = VaultClient(decode_files=True)
    head_version = client.get_version(TENTACLES_VAULT_SECRET_UUID)
    return head_version["value"]["JUGGLER_OAUTH_TOKEN"]


def parse_args() -> argparse.Namespace:
    """
        Parses CLI arguments.
    """
    parser = argparse.ArgumentParser()
    parser.add_argument("--stage",
                        choices=["testing", "production"],
                        required=True,
                        help="Installation stage type")

    parser.add_argument("--config-dir",
                        default=os.path.join("..", "..", "conf"),
                        help="Path to config directory where file named `stage`.yaml expected")

    parser.add_argument("--harvesters-snapshots-freshness", action="store_true", default=False,
                        help='Configure harvesters snapshots freshness checks'
                        )
    parser.add_argument("--reallocation", action="store_true", default=False,
                        help='Configure reallocation checks'
                        )
    parser.add_argument("--redeployment", action="store_true", default=False,
                        help='Configure redeployment checks'
                        )
    parser.add_argument("--slo", action="store_true", default=False,
                        help='Configure SLO checks',
                        )
    parser.add_argument("--golovan-alerts", action="store_true", default=False,
                        help='Replace golovan alerts',
                        )
    parser.add_argument("--tentacles-containers", action="store_true", default=False,
                        help='Configure Tentacles containers checks',
                        )
    parser.add_argument("--no-common", action="store_true", default=False,
                        help='Do not configure common checks',
                        )
    parser.add_argument("--all-backend", action="store_true", default=False,
                        help='Apply all backend checks',
                        )
    parser.add_argument("--apply-changes",
                        action="store_true",
                        required=False,
                        help="If given, apply changes to API")
    parser.add_argument("--juggler-oauth-token",
                        required=False,
                        default=get_juggler_oauth_token_from_vault(),
                        help="Override JUGGLER_OAUTH_TOKEN that is stored in Ya Vault")
    return parser.parse_args()


def run_upsert_on_check(_check: juggler_sdk.Check, _api: juggler_sdk.JugglerApi) -> None:
    """
         Prints check details and applies check to Juggler API with
         `upsert_check` SDK method. Prints changes in checks' config
         if they are returned from API.
    """
    try:
        header = f"  * host={_check.host} & service={check.service} ({check.description})" \
            if check.description \
            else f"  * host={_check.host} & service={check.service}"
        print(header)
        print(
            (f"    https://juggler.yandex-team.ru/check_details/?"
             f"host={_check.host}&service={_check.service}"))
        upsert_result = _api.upsert_check(_check)
        if upsert_result.changed:
            print("    Changes:")
            pprint(upsert_result.diff.to_dict(),
                   indent=2)
            print()
        else:
            print("    No changes\n")
    except juggler_sdk.errors.JugglerError as _e:
        print(str(_e))
        raise SystemExit(1)


if __name__ == "__main__":
    args = parse_args()

    stage_config_file_name = f"{args.stage}.yaml"
    stage_config_file_path = os.path.join(args.config_dir, stage_config_file_name)
    try:
        file_config = read_yaml_file(stage_config_file_path)
    except Exception:
        print(f"Can not load YAML config file {stage_config_file_path!r}")
        raise

    check_group_names_args_mapping = {
        "harvesters_snapshots_freshness": args.harvesters_snapshots_freshness,
        "reallocation": args.reallocation,
        "redeployment": args.redeployment,
        "slo": args.slo,
        "tentacles_containers": args.tentacles_containers
    }

    juggler_checks_factory = JugglerChecksFactory(file_config=file_config)

    dry_run = not args.apply_changes

    with juggler_sdk.JugglerApi(JUGGLER_API_URL, oauth_token=args.juggler_oauth_token, dry_run=dry_run) as api:
        group_checks = []
        if args.all_backend:
            group_checks = BACKEND_CHECK_GROUP_NAMES
        else:
            if not args.no_common:
                group_checks.append("common")
            for check_group_name, value in check_group_names_args_mapping.items():
                if value:
                    group_checks.append(check_group_name)

        for check_group_name in group_checks:
            print(f"Updating {check_group_name!r} checks (dry_run={dry_run})")
            for check in juggler_checks_factory.get_checks(stage=args.stage, checks_group=check_group_name):
                run_upsert_on_check(check, api)

    if args.golovan_alerts or args.all_backend:
        prefix = f"tentacles_{args.stage}"
        alerts = golovan.generate_alerts(prefix, ctype=args.stage, file_config=file_config)
        print(f"Updating golovan alerts (with juggler checks) (dry_run={dry_run})\n")
        print(json.dumps(alerts, indent=4, sort_keys=True))
        if not dry_run:
            golovan.replace(prefix, alerts)
