import argparse
import os
import signal
import tarfile
import time
import shutil

from flask import Flask, json

from infra.rtc_sla_tentacles.backend.lib.tentacle_agent import const
from infra.rtc_sla_tentacles.backend.lib.tentacle_agent.tentacle_state import TentacleState


def parse_arguments():
    parser = argparse.ArgumentParser(description="RTC SLA Tentacles web server.")

    web_server_settings_group = parser.add_argument_group("web_server_settings_group", "Web server settings")
    web_server_settings_group.add_argument("--host",
                                           required=True,
                                           help="FQDN/IP to listen on",
                                           dest="HOST")
    web_server_settings_group.add_argument("--port",
                                           type=int,
                                           required=True,
                                           help="Port to listen on",
                                           dest="PORT")

    timestamp_files_settings_group = parser.add_argument_group("timestamp_files_settings_group",
                                                               "Timestamp files settings")
    timestamp_files_settings_group.add_argument("--timestamp-resource-file-path",
                                                required=True,
                                                help="Absolute path to 'timestamp resource' file",
                                                dest="TIMESTAMP_RESOURCE_FILE_PATH")
    timestamp_files_settings_group.add_argument("--timestamp-resource-archive-file-path",
                                                required=False,
                                                default=None,
                                                help=("If set, extracts provided archive to 'timestamp resource' "
                                                      "directory"),
                                                dest="TIMESTAMP_RESOURCE_ARCHIVE_FILE_PATH")
    timestamp_files_settings_group.add_argument("--timestamp-prepare-finish-time-file-path",
                                                required=True,
                                                help="Absolute path to 'timestamp resource prepare finish time' file",
                                                dest="TIMESTAMP_PREPARE_FINISH_TIME_FILE_PATH")
    timestamp_files_settings_group.add_argument("--timestamp-self-shutdown-time-file-path",
                                                required=True,
                                                help="Absolute path to 'self shutdown time' file",
                                                dest="TIMESTAMP_SELF_SHUTDOWN_TIME_FILE_PATH")

    juggler_settings_group = parser.add_argument_group("juggler_settings_group", "Juggler settings")
    juggler_settings_group.add_argument("--juggler-crit-age",
                                        type=int,
                                        default=3000,
                                        required=False,
                                        help=("Response CRIT to Juggler if timestamp is older than "
                                              "JUGGLER_CRIT_AGE seconds"),
                                        dest="JUGGLER_CRIT_AGE")
    juggler_settings_group.add_argument("--juggler-ok-code",
                                        type=int,
                                        default=200,
                                        required=False,
                                        help="Juggler OK HTTP code",
                                        dest="JUGGLER_OK_CODE")
    juggler_settings_group.add_argument("--juggler-crit-code",
                                        type=int,
                                        default=418,
                                        required=False,
                                        help="Juggler CRIT HTTP code",
                                        dest="JUGGLER_CRIT_CODE")
    args = parser.parse_args()
    _web_server_settings = {
        "host": args.HOST,
        "port": args.PORT
    }
    _timestamp_files_settings = {
        "resource_file_path": args.TIMESTAMP_RESOURCE_FILE_PATH,
        "resource_archive_file_path": args.TIMESTAMP_RESOURCE_ARCHIVE_FILE_PATH,
        "prepare_finish_time_file_path": args.TIMESTAMP_PREPARE_FINISH_TIME_FILE_PATH,
        "self_shutdown_time_file_path": args.TIMESTAMP_SELF_SHUTDOWN_TIME_FILE_PATH,
    }
    _juggler_settings = {
        "crit_age": args.JUGGLER_CRIT_AGE,
        "ok_code": args.JUGGLER_OK_CODE,
        "crit_code": args.JUGGLER_CRIT_CODE
    }
    return _web_server_settings, _timestamp_files_settings, _juggler_settings


def get_flask_app(app_name, _tentacle_state):
    _app = Flask(app_name)

    @_app.route("/")
    def index():
        timestamps = _tentacle_state.get_timestamps()
        response = app.response_class(response=json.dumps(timestamps),
                                      status=200,
                                      mimetype="application/json")
        return response

    @_app.route("/juggler")
    def juggler():
        description, http_code = _tentacle_state.get_juggler_timestamp_age_monitoring()
        response = app.response_class(response=description,
                                      status=http_code)
        return response

    @_app.route("/golovan")
    def golovan():
        golovan_data = _tentacle_state.get_golovan_data()
        response = app.response_class(response=json.dumps(golovan_data),
                                      status=200,
                                      mimetype="application/json")
        return response

    return _app


def extract_resource_archive(resource_archive_file_path, resource_file_path):
    resource_dir, _ = os.path.split(resource_file_path)
    with tarfile.open(resource_archive_file_path, "r") as tar:
        tar.extractall(path=resource_dir)


def manage_agent_resource():
    # NOTE(rocco66): we should check '/data' volume, so let's copy resource there
    if os.path.exists(const.AGENT_RESOURCE_NAME):
        shutil.copyfile(const.AGENT_RESOURCE_NAME, timestamp_files_settings["resource_file_path"])
    elif timestamp_files_settings["resource_archive_file_path"]:
        extract_resource_archive(resource_archive_file_path=timestamp_files_settings["resource_archive_file_path"],
                                 resource_file_path=timestamp_files_settings["resource_file_path"])


if __name__ == "__main__":
    web_server_settings, timestamp_files_settings, juggler_settings = parse_arguments()

    def _handle_exit(_sig, _frame):
        with open(timestamp_files_settings["self_shutdown_time_file_path"], "w") as f:
            f.write(str(int(time.time())))
        raise SystemExit

    signal.signal(signal.SIGINT, _handle_exit)
    signal.signal(signal.SIGTERM, _handle_exit)

    manage_agent_resource()

    tentacle_state = TentacleState(binary_start_ts=int(time.time()),
                                   timestamp_files_settings=timestamp_files_settings,
                                   juggler_settings=juggler_settings)

    app = get_flask_app("tentacles", tentacle_state)
    app.run(host=web_server_settings["host"], port=web_server_settings["port"])
