import argparse
import json
import logging

from dataclasses import asdict

from infra.rtc_sla_tentacles.backend.lib.config.interface import ConfigInterface
from infra.rtc_sla_tentacles.backend.lib.config.raw_config_storage import RawConfigStorage
from infra.rtc_sla_tentacles.backend.lib.yp_lite.pods_manager import YpLitePodsManager
from infra.rtc_sla_tentacles.backend.lib.util import read_yaml_file


def parse_args() -> argparse.Namespace:
    """
        Parses CLI arguments.
    """
    parser = argparse.ArgumentParser()
    parser.add_argument("-c", "--config", required=True, help="Path to YAML config file", metavar="path/to.yaml")
    parser.add_argument("--nanny-service-id", required=True, help="Nanny service ID")
    parser.add_argument("--yp-cluster", required=True, help="YP cluster")
    parser.add_argument("-r", "--restrict_nodes_to_specified_in_podset", action="store_true", default=False,
                        help="Narrow YP nodes to nodes specified in podset's '/spec/node_filter'")
    parser.add_argument("--data-source", choices=["api", "clickhouse"], required=True, help="Pods statuses source")
    parser.add_argument("-v", "--verbose", action="store_true", default=False, help="Print pods data")

    subparsers = parser.add_subparsers(dest="subparser_name", help=("Subcommands that manage pods, Nanny instances "
                                       "or podset's node filter"))

    parser_pods = subparsers.add_parser("pods", help="Managing pods")
    parser_pods.add_argument("--action", choices=["update"], required=True)

    parser_remove_pod = subparsers.add_parser("remove_pod", help="Remove pod")
    parser_remove_pod.add_argument("pod_id", type=str, nargs='+')

    parser_nanny_instances = subparsers.add_parser("nanny-instances", help="Managing Nanny instances")
    parser_nanny_instances.add_argument("--action", choices=["update"], required=True)

    parser_node_filter = subparsers.add_parser("node-filter", help="Manage podset's node filter")
    parser_node_filter.add_argument("--action", choices=["disable", "get", "set"], required=True)
    parser_node_filter.add_argument("--percent-by-rack", type=int, default=0)

    return parser.parse_args()


def prepare_config(_args) -> RawConfigStorage:
    config = read_yaml_file(_args.config)
    # fix logging
    config["logging"]["formatters"]["simple"] = {
        "format": "%(asctime)s.%(msecs)03d [%(levelname)-1s] [%(name)-10s]   %(message)s",
        "datefmt": "%Y-%m-%d %H:%M:%S",
    }
    config["logging"]["handlers"].clear()
    config["logging"]["handlers"]["common"] = {
        "class": "logging.StreamHandler",
        "formatter": "simple",
    }
    config["logging"]["loggers"].clear()
    config["logging"]["loggers"][""] = {
        "level": "DEBUG",
        "handlers": ["common"],
    }
    config["logging"]["loggers"]["YpClient"] = {
        "level": "DEBUG",
        "handlers": ["common"],
        "propagate": False,
    }

    _storage = RawConfigStorage(
        cli_args={
            "SECRETS_DIRECTORY": None,
            "MY_DATACENTER_NAME": None,
            "MONGO_DB_NAME": None,
            "CH_DB_NAME": None,
        },
        file_cfg=config,
    )
    return _storage


def manage_pods(_pods_manager: YpLitePodsManager, _args: argparse.Namespace):
    if args.action == "update":
        print("\n\nUpdating pods")
        pods_manager.update_pods()


def manage_nanny_instances(_pods_manager: YpLitePodsManager, _args: argparse.Namespace):
    if args.action == "update":
        print("\n\nUpdating Nanny instances")
        pods_manager.update_nanny_instances()


def manage_node_filter(_pods_manager: YpLitePodsManager, _args: argparse.Namespace):
    if args.action == "disable":
        pods_manager.set_empty_node_filter()
    elif args.action == "get":
        podset_node_filter = pods_manager.get_podset_node_filter()
        print("\n\nCurrent podset's node filter: %s" % podset_node_filter)
    elif args.action == "set":
        if not 0 < args.percent_by_rack <= 100:
            raise ValueError("'Percent by rack' has invalid value.")
        pods_manager.set_node_filter_rack_string_comparison(args.percent_by_rack)


if __name__ == "__main__":
    args = parse_args()

    storage = prepare_config(args)
    config_interface = ConfigInterface(storage)

    with YpLitePodsManager(
        args.nanny_service_id, args.yp_cluster,
        restrict_nodes_to_specified_in_podset=args.restrict_nodes_to_specified_in_podset,
        config_interface=config_interface,
        init_yp_client=True,
        logger=logging.getLogger(),
        init_clickhouse_client=True,
        init_nanny_service_repo_client=True,
        init_nanny_podsets_service_stub=True,
    ) as pods_manager:
        if args.data_source == "api":
            pods_manager.load_data_from_api()
        elif args.data_source == "clickhouse":
            pods_manager.load_pods_from_clickhouse()

        if args.verbose:
            print("\n\nPods data:")
            # noinspection PyProtectedMember
            for pod_data in pods_manager._pods_data:
                print(json.dumps(asdict(pod_data), indent=4))

        stats = pods_manager.get_stats()
        print("\n\nStats:")
        print(json.dumps(stats, indent=4))

        if args.subparser_name == "pods":
            manage_pods(pods_manager, args)
        elif args.subparser_name == "remove_pod":
            for pod_id in args.pod_id:
                pods_manager.remove_pod(pod_id)
        elif args.subparser_name == "nanny-instances":
            manage_nanny_instances(pods_manager, args)
        elif args.subparser_name == "node-filter":
            manage_node_filter(pods_manager, args)
