import dataclasses
import time
import typing

from infra.rtc_sla_tentacles.backend.lib.metrics import metrics_provider
from infra.rtc_sla_tentacles.backend.lib.api.proto import allocation_zone_features_pb2
from infra.rtc_sla_tentacles.backend.lib.config import utils as config_utils


@dataclasses.dataclass()
class AllocationZonesMetrics:
    metrics: typing.Dict[str, metrics_provider.AllZoneMetrics]
    ts: int


def is_metrics_fresh(cache_ts):
    return cache_ts + metrics_provider.MONITORING_DATA_TTL >= int(time.time())


def _bool_to_feature_status(status):
    return (
        allocation_zone_features_pb2.EFeatureStatus.OK
        if status
        else allocation_zone_features_pb2.EFeatureStatus.BROKEN
    )


def _fill_availability(availability_metrics, availability_proto):
    availability_proto.status = _bool_to_feature_status(availability_metrics.is_slo_ok())
    availability_proto.ok_count = availability_metrics.number_of_available_tentacles
    availability_proto.total_count = availability_metrics.number_of_not_excluded_tentacles


def _fill_redeploy(redeploy_metrics, redeploy_proto):
    redeploy_proto.status = _bool_to_feature_status(redeploy_metrics.is_slo_ok())
    redeploy_proto.ok_count = redeploy_metrics.number_of_tentacles_with_fresh_ts_resource
    redeploy_proto.total_count = redeploy_metrics.number_of_not_excluded_tentacles

    redeploy_proto.last_success_redeploy_timestamp = redeploy_metrics.last_success_redeployment_end_ts
    redeploy_proto.current_redeploy_timestamp = redeploy_metrics.current_redeployment_start_ts


def _fill_reallocation(reallocation_metrics, reallocation_proto, reallocation_settings):
    if reallocation_settings and reallocation_settings.get("monitoring"):
        reallocation_proto.enable = True
        reallocation_proto.status = _bool_to_feature_status(reallocation_metrics.is_slo_ok())
        reallocation_proto.last_success_reallocation_timestamp = \
            reallocation_metrics.last_success_reallocation_end_ts
        reallocation_proto.current_reallocation_timestamp = reallocation_metrics.current_reallocation_start_ts
    else:
        reallocation_proto.enable = False


def _fill_scheduling_errors(allocation_zone_id, scheduling_errors_metrics, scheduling_errors_proto):
    if config_utils.is_daemonset_location(allocation_zone_id):
        scheduling_errors_proto.enable = True
        scheduling_errors_proto.status = _bool_to_feature_status(scheduling_errors_metrics.is_slo_ok())
        scheduling_errors_proto.scheduling_errors_on_up_nodes_count = \
            scheduling_errors_metrics.scheduling_errors_on_up_nodes_count
        scheduling_errors_proto.total_up_nodes_count = scheduling_errors_metrics.total_up_nodes_count
    else:
        scheduling_errors_proto.enable = False


def fill_allocation_zone_features(
    allocation_zones_metrics: AllocationZonesMetrics,
    allocation_zone_proto: allocation_zone_features_pb2.TAllocationZoneFeatures,
    allocation_zone_id: str,
    allocation_zone_config: dict,
):
    allocation_zone_proto.allocation_zone = allocation_zone_id
    allocation_zone_metrics = allocation_zones_metrics.metrics.get(allocation_zone_id)
    if not allocation_zone_metrics:
        allocation_zone_proto.data_freshness.status = allocation_zone_features_pb2.EFeatureStatus.BROKEN
        return

    allocation_zone_proto.data_freshness.data_timestamp = allocation_zones_metrics.ts
    allocation_zone_proto.data_freshness.status = _bool_to_feature_status(
        is_metrics_fresh(allocation_zones_metrics.ts)
    )
    _fill_availability(allocation_zone_metrics.availability, allocation_zone_proto.availability)
    _fill_redeploy(allocation_zone_metrics.redeployed_on_time, allocation_zone_proto.redeploy)

    reallocation_settings = allocation_zone_config.get("reallocation_settings")
    _fill_reallocation(allocation_zone_metrics.reallocation, allocation_zone_proto.reallocation, reallocation_settings)

    _fill_scheduling_errors(allocation_zone_id, allocation_zone_metrics.scheduling_errors,
                            allocation_zone_proto.scheduling_errors)
