import typing

from infra.rtc_sla_tentacles.backend.lib.clickhouse.client import ClickhouseClient
from infra.rtc_sla_tentacles.backend.lib.clickhouse import database
from infra.rtc_sla_tentacles.backend.lib.api.proto import allocation_zone_details_pb2, filters_pb2
from infra.rtc_sla_tentacles.backend.lib.metrics import metrics_provider


POD_COLUMNS = [
    "walle_state",
    "walle_operation_state",
    "excluded_from_slo_by_walle",
    "excluded_from_slo_by_allocator",
    "yp_scheduling_error",
    "yp_node_hfsm_state",
    "is_hq_data_present",
    "is_juggler_data_present",
    "monitoring_timestamp_age_present",
    "monitoring_timestamp_age_visible",
    "monitoring_timestamp_age_reachable",
    "slo_availablilty_excluded",
    "slo_availablilty_available",
    "slo_redeployment_excluded",
    "slo_redeployment_fresh_ts_resource",
]

ITERATION_COLUMNS = [
    "juggler_data_freshness",
    "hq_data_freshness",
    "walle_data_freshness",
]

WALLE_SLICES_COLUMNS = [
    "pod_id",
    "fqdn",
    "walle_project",
    "rtc_stage",
    "walle_operation_state",
    "walle_state",
    "walle_scenario_id",
    "walle_ticket",
]


def _fill_slices(pipeline):
    pipeline.walle_node.slices.extend([
        allocation_zone_details_pb2.TSlice(
            filter=filters_pb2.EPodFilter.EXCLUDED_DECOMMISSIONED_NODES,
            columns=WALLE_SLICES_COLUMNS,
        ),
        allocation_zone_details_pb2.TSlice(
            filter=filters_pb2.EPodFilter.EXCLUDED_PROBATION_OR_FREE_NODES,
            columns=WALLE_SLICES_COLUMNS,
        ),
    ])
    pipeline.allocator_node.slices.extend([
        allocation_zone_details_pb2.TSlice(
            filter=filters_pb2.EPodFilter.PODS_SCHEDULING_ERRORS,
            columns=["pod_id", "fqdn", "walle_project", "rtc_stage", "yp_node_hfsm_state", "yp_scheduling_error"]
        ),
        allocation_zone_details_pb2.TSlice(
            filter=filters_pb2.EPodFilter.UNUSED_YP_NODES,
            columns=[
                "pod_id", "fqdn", "walle_project", "yp_node_hfsm_state", "yp_scheduling_error",
                 "juggler_state_kind", "juggler_status",
            ]
        ),
    ])
    pipeline.juggler_node.slices.extend([
        allocation_zone_details_pb2.TSlice(
            filter=filters_pb2.EPodFilter.EXCLUDED_NOT_PRESENT,
            columns=["pod_id", "fqdn", "walle_project", "rtc_stage"]
        ),
        allocation_zone_details_pb2.TSlice(
            filter=filters_pb2.EPodFilter.EXCLUDED_NOT_VISIBLE,
            columns=["pod_id", "fqdn", "walle_project", "juggler_state_kind"]
        ),
    ])

    pipeline.availability_node.slices.extend([
        allocation_zone_details_pb2.TSlice(
            filter=filters_pb2.EPodFilter.EXCLUDED_NOT_REACHABLE,
            columns=["pod_id", "fqdn", "walle_project", "juggler_status", "juggler_state_kind", "juggler_description"]
        ),
    ])


def _get_pods_data(client, allocation_zone_id):
    pod_columns_str = ", ".join(POD_COLUMNS)
    return client.select(
        f"""
        SELECT {pod_columns_str}, COUNT() as count
        FROM {database.Tentacle.table_name()}
        WHERE ts={database.get_last_timestamp()} AND nanny_service_name='{allocation_zone_id}'
        GROUP BY {pod_columns_str}
        """
    )


def _get_iteration(client):
    iteration_columns_str = ", ".join(ITERATION_COLUMNS)
    return next(iter(client.select(
        f"""
        SELECT {iteration_columns_str}
        FROM {database.Iteration.table_name()}
        WHERE ts={database.get_last_timestamp()}
        LIMIT 1
        """
    )))


def fill_allocation_zone_pipeline(
    client: ClickhouseClient,
    allocation_zone_id: str,
    protobuf_result: allocation_zone_details_pb2.TAllocationZoneDetailsResult,
    yp_unused_nodes_metrics: typing.Optional[metrics_provider.MetricsTentaclesYpUnusedNodes],
):
    pipeline = protobuf_result.pipeline
    # NOTE(rocco66): for walle message initializing
    pipeline.walle_node.excluded_probation_or_free_nodes_count = 0

    for row in _get_pods_data(client, allocation_zone_id):
        if row.walle_state in ["probation", "free"]:
            pipeline.walle_node.excluded_probation_or_free_nodes_count += row.count
        if row.walle_operation_state and row.walle_operation_state.name == "decommissioned":
            pipeline.walle_node.excluded_decommissioned_nodes_count += row.count

        pipeline.allocator_node.total_pods_count += row.count
        if row.yp_node_hfsm_state == "up" and row.yp_scheduling_error:
            pipeline.allocator_node.pods_scheduling_errors += row.count

        if row.is_hq_data_present:
            pipeline.hq_node.total_pods += row.count

        if row.is_juggler_data_present:
            pipeline.juggler_node.total_pods += row.count
        excluded_and_ignored = row.excluded_from_slo_by_walle or row.excluded_from_slo_by_allocator
        if not row.monitoring_timestamp_age_present and not excluded_and_ignored:
            pipeline.juggler_node.excluded_not_present += row.count
        if not row.monitoring_timestamp_age_visible and not excluded_and_ignored:
            pipeline.juggler_node.excluded_not_visible += row.count

        if not row.slo_availablilty_excluded:
            pipeline.availability_node.total_count += row.count
            if row.slo_availablilty_available:
                pipeline.availability_node.ok_count += row.count
            if not row.monitoring_timestamp_age_reachable:
                pipeline.availability_node.excluded_not_reachable += row.count

        if not row.slo_redeployment_excluded:
            pipeline.redeployment_node.total_count += row.count
            if row.slo_redeployment_fresh_ts_resource:
                pipeline.redeployment_node.ok_count += row.count

    if yp_unused_nodes_metrics:
        pipeline.allocator_node.total_nodes_count = yp_unused_nodes_metrics.total_nodes_count
        pipeline.allocator_node.unused_nodes_count = yp_unused_nodes_metrics.unused_nodes_count

    iteration_result = _get_iteration(client)
    protobuf_result.freshness.walle_ts = int(iteration_result.walle_data_freshness.timestamp())
    protobuf_result.freshness.juggler_ts = int(iteration_result.juggler_data_freshness.timestamp())
    protobuf_result.freshness.hq_ts = int(iteration_result.hq_data_freshness.timestamp())

    _fill_slices(pipeline)
