import typing

from infra.rtc_sla_tentacles.backend.lib.api.proto import pods_list_pb2, filters_pb2
from infra.rtc_sla_tentacles.backend.lib.clickhouse import database, client
from infra.rtc_sla_tentacles.backend.lib.api import utils
from infra.rtc_sla_tentacles.backend.lib.metrics import metrics_provider

DEFAULT_PODS_LIMIT = 1000
MAX_PODS_LIMIT = 1000


def _get_conditions(request, yp_unused_nodes_metrics):
    input_ts = request.ts
    if input_ts:
        target_ts = f"'{database.to_clickhouse_datetime(input_ts)}'"
    else:
        target_ts = f"(SELECT MAX(ts) FROM {database.Iteration.table_name()})"

    if request.filter == filters_pb2.EPodFilter.EXCLUDED_DECOMMISSIONED_NODES:
        filters_str = "walle_operation_state = 'decommissioned'"
    elif request.filter == filters_pb2.EPodFilter.EXCLUDED_PROBATION_OR_FREE_NODES:
        filters_str = "walle_state IN ['probation', 'free']"
    elif request.filter == filters_pb2.EPodFilter.PODS_SCHEDULING_ERRORS:
        filters_str = "notEmpty(yp_scheduling_error) AND yp_node_hfsm_state = 'up'"
    elif request.filter == filters_pb2.EPodFilter.UNUSED_YP_NODES:
        if yp_unused_nodes_metrics and yp_unused_nodes_metrics.unused_nodes_count:
            unused_nodes_target_chunk = yp_unused_nodes_metrics.unused_nodes[:MAX_PODS_LIMIT]
        else:
            unused_nodes_target_chunk = []
        unused_nodes_str_list = ", ".join(f"'{n}'" for n in unused_nodes_target_chunk)
        filters_str = f"fqdn IN [{unused_nodes_str_list}]"
    elif request.filter == filters_pb2.EPodFilter.EXCLUDED_NOT_PRESENT:
        filters_str = "NOT excluded_from_slo_by_walle AND NOT monitoring_timestamp_age_present"
    elif request.filter == filters_pb2.EPodFilter.EXCLUDED_NOT_VISIBLE:
        filters_str = "NOT excluded_from_slo_by_walle AND NOT monitoring_timestamp_age_visible"
    elif request.filter == filters_pb2.EPodFilter.EXCLUDED_NOT_REACHABLE:
        filters_str = "NOT slo_availablilty_excluded AND NOT monitoring_timestamp_age_reachable"
    else:
        filters_str = ""

    conditions = f"ts = {target_ts} AND nanny_service_name = '{request.allocation_zone_id}'"
    if filters_str:
        conditions += f" AND {filters_str}"
    return conditions


def _get_pods_query(request, columns, conditions_clause):
    columns_by_comma = ", ".join(columns)

    limit = min(request.limit or DEFAULT_PODS_LIMIT, MAX_PODS_LIMIT)
    offset = request.offset

    return f"""
        SELECT {columns_by_comma}
        FROM {database.Tentacle.table_name()}
        WHERE {conditions_clause}
        ORDER BY fqdn ASC
        LIMIT {limit} OFFSET {offset}
    """


def get_pods_list(request: pods_list_pb2.TPodsListRequest, config_interface,
                  yp_unused_nodes_metrics: typing.Optional[metrics_provider.MetricsTentaclesYpUnusedNodes],
                  ) -> pods_list_pb2.TPodsListResponse:
    columns = request.columns
    if "fqdn" not in columns:
        columns.append("fqdn")

    response = pods_list_pb2.TPodsListResponse()
    conditions_clause = _get_conditions(request, yp_unused_nodes_metrics)

    clickhouse_client = client.ClickhouseClient(config_interface)
    response.result.total_count = clickhouse_client.count(database.Tentacle, conditions_clause)
    tentacles = response.result.tentacles

    for db_row in clickhouse_client.select(_get_pods_query(request, columns, conditions_clause)):
        tentacle = tentacles.add()
        for column in columns:
            raw_val = getattr(db_row, column)
            if raw_val is not None:
                setattr(tentacle, column, utils.from_db_to_proto(raw_val))
    return response
