import logging
import random

from infi.clickhouse_orm.database import Database
from infra.rtc_sla_tentacles.backend.lib.funccall_stats_server import server as stat_server


class ClickhouseClient:

    def __init__(self, config_interface, leader_only=False, force_node=None):
        self._clickhouse_params = config_interface.get_api_config()["clickhouse_parameters"]
        self._force_node = force_node
        self._leader_only = leader_only
        self._db = None

    def _is_replication_cluster(self):
        total_hosts_count = (
            len(self._clickhouse_params["hosts"]["in_my_dc"]) + len(self._clickhouse_params["hosts"]["in_other_dcs"])
        )
        return total_hosts_count > 1

    def _get_dc(self):
        if self._force_node:
            yield self._force_node
            return
        hosts_in_my_dc = self._clickhouse_params["hosts"]["in_my_dc"]
        hosts_in_other_dcs = self._clickhouse_params["hosts"]["in_other_dcs"]
        random.shuffle(hosts_in_my_dc)
        for _host in hosts_in_my_dc:
            yield _host
        random.shuffle(hosts_in_other_dcs)
        for _host in hosts_in_other_dcs:
            yield _host

    def _get_db(self):
        last_exc = None
        for host in self._get_dc():
            fqdn = host["fqdn"]
            port = host["port"]
            logging.debug(f"Connecting to ClickHouse server {fqdn!r}...")
            try:
                db = Database(
                    self._clickhouse_params["database"],
                    db_url=f"https://{fqdn}:{port}",
                    username=self._clickhouse_params["user"],
                    password=self._clickhouse_params["password"],
                    timeout=(self._clickhouse_params["connect_timeout"], self._clickhouse_params["read_timeout"]),
                    verify_ssl_cert=bool(self._clickhouse_params.get("ca_certs", True)),
                )
            except Exception as exc:
                last_exc = exc
                logging.exception(f"Failed to connect to ClickHouse server {fqdn!r}:")
                continue
            if self._leader_only and self._is_replication_cluster():
                if not int(db.raw("SELECT is_leader FROM system.replicas LIMIT 1")):
                    continue
            return db
        if last_exc:
            raise last_exc
        raise Exception("Failed to connect to all configured ClickHouse servers")

    def raw(self, *args, **kwargs):
        if not self._db:
            self._db = self._get_db()
        with stat_server.clickhouse_timing():
            return self._db.raw(*args, **kwargs)

    def insert(self, *args, **kwargs):
        if not self._db:
            self._db = self._get_db()
        with stat_server.clickhouse_timing():
            return self._db.insert(*args, **kwargs)

    def select(self, *args, **kwargs):
        if not self._db:
            self._db = self._get_db()
        with stat_server.clickhouse_timing():
            return self._db.select(*args, **kwargs)

    def count(self, *args, **kwargs):
        if not self._db:
            self._db = self._get_db()
        with stat_server.clickhouse_timing():
            return self._db.count(*args, **kwargs)

    def create_table(self, *args, **kwargs):
        if not self._db:
            self._db = self._get_db()
        return self._db.create_table(*args, **kwargs)
