import datetime
from enum import Enum, EnumMeta  # type: ignore
from itertools import product
from infi.clickhouse_orm.models import Model, ModelBase
from infi.clickhouse_orm.fields import (
    StringField,
    NullableField,
    ArrayField,
    DateTimeField,
    UInt8Field,
    UInt64Field,
    Enum8Field,
)


_CLICKHOUSE_DATETIME_FORMAT = "%Y-%m-%d %H:%M:%S"


class DefaultEnumMeta(EnumMeta):
    default = object()

    def __call__(cls, value=default, *args, **kwargs):
        if value is DefaultEnumMeta.default:
            return next(iter(cls))
        return super().__call__(value, *args, **kwargs)

    def __getitem__(cls, item):
        if item in cls._member_map_:
            return cls._member_map_[item]
        return next(iter(cls))


class DefaultEnum(Enum, metaclass=DefaultEnumMeta):
    pass


nanny_service_name_values = [
    "OTHER",
    "rtc_sla_tentacles_production_sas",
    "rtc_sla_tentacles_production_vla",
    "rtc_sla_tentacles_production_man",
    "rtc_sla_tentacles_production_msk",
    "yp_rtc_sla_tentacles_production_sas",
    "yp_rtc_sla_tentacles_production_vla",
    "yp_rtc_sla_tentacles_production_man",
    "yp_rtc_sla_tentacles_production_man_pre",
    "yp_daemonset_sas_rtc_sla_tentacles_production",
    "yp_daemonset_vla_rtc_sla_tentacles_production",
    "yp_daemonset_man_rtc_sla_tentacles_production",
    "yp_daemonset_iva_rtc_sla_tentacles_production",
    "yp_daemonset_myt_rtc_sla_tentacles_production",
    "yp_daemonset_xdc_rtc_sla_tentacles_production",
    "yp_daemonset_man_pre_rtc_sla_tentacles_production",
    "yp_daemonset_sas_test_rtc_sla_tentacles_production",
]
nanny_service_name_enum = DefaultEnum("nanny_service_name_enum", nanny_service_name_values)

walle_health_fail_prefixes = ["unreachable", "ssh", "meta", "walle_meta", "memory", "disk", "link", "bmc", "gpu",
                              "cpu_caches", "reboots", "tainted_kernel", "cpu", "overheat", "cpu_capping", "fs_check",
                              "switch", "rack"]
walle_health_fail_suffixes = ["failed", "suspected", "missing", "invalid", "void", "staled", "unsure"]
walle_health_fail_reasons = ['OTHER', "no", "ok", "failure"] + walle_health_fail_prefixes + \
                            [f"{i}-{j}" for i, j in product(walle_health_fail_prefixes, walle_health_fail_suffixes)]
walle_health_fail_enum = DefaultEnum("walle_health_fail_enum",  # type: ignore
                                     walle_health_fail_reasons,
                                     start=-50)

walle_operation_state_values = ["operation", "decommissioned"]
walle_operation_state_enum = DefaultEnum("walle_operation_state_enum", walle_operation_state_values)  # type: ignore


juggler_state_kind_values = ["other", "actual", "downtime_force_ok", "flapping", "no_data_force_crit"]
juggler_state_kind_enum = DefaultEnum("juggler_state_kind", juggler_state_kind_values)  # type: ignore

juggler_status_values = ["other", "ok", "warn", "crit"]
juggler_status_enum = DefaultEnum("juggler_status", juggler_status_values)  # type: ignore

juggler_description_values = [
    "other",
    "ok",
    "timestamp_too_old",
    "resource_error",
    "icmp_down",
    "invalid_hostname",
    "http_connect_error",
    "no_data",
]
juggler_description_enum = DefaultEnum("juggler_description", juggler_description_values)  # type: ignore

rtc_stage_values = ["production", "prestable", "experiment"]
rtc_stage_enum = DefaultEnum("rtc_stage_enum", rtc_stage_values)


TABLE_OBJECT_MAPPING = {}


class TablenameToClassIndex(ModelBase):

    def __new__(cls, name, bases, attrs):
        new_class = super(TablenameToClassIndex, cls).__new__(cls, str(name), bases, attrs)
        TABLE_OBJECT_MAPPING[new_class.table_name()] = new_class
        return new_class


class TentaclesModel(Model, metaclass=TablenameToClassIndex):

    history_depth = 4 * 24 * 60 * 60
    date_col = "ts"
    order_by = ["ts"]


class Tentacle(TentaclesModel):

    @classmethod
    def table_name(cls):
        return "tentacles_results"

    ts = DateTimeField()
    pod_id = StringField()
    fqdn = StringField()
    tentacles_location = StringField()
    rtc_stage = NullableField(Enum8Field(rtc_stage_enum))
    deploy_gencfg = NullableField(UInt8Field())
    deploy_yp_lite = NullableField(UInt8Field())

    yp_scheduling_error = StringField()
    yp_node_hfsm_state = StringField()

    is_juggler_data_present = UInt8Field()
    juggler_status = NullableField(Enum8Field(juggler_status_enum))
    juggler_state_kind = NullableField(Enum8Field(juggler_state_kind_enum))
    juggler_description = NullableField(Enum8Field(juggler_description_enum))

    nanny_service_name = Enum8Field(nanny_service_name_enum)
    hq_id = NullableField(StringField())
    hq_hostname = NullableField(StringField())
    is_hq_data_present = UInt8Field()

    excluded_from_slo_by_walle = UInt8Field()
    excluded_from_slo_by_allocator = UInt8Field()
    walle_project = StringField()
    walle_location_short_datacenter_name = NullableField(StringField())
    walle_operation_state = NullableField(Enum8Field(walle_operation_state_enum))
    walle_state = StringField()
    walle_status = StringField()
    walle_health_status = NullableField(StringField())
    walle_health_fail_reasons = ArrayField(Enum8Field(walle_health_fail_enum))
    walle_scenario_id = UInt64Field()
    walle_ticket = StringField()

    # See https://wiki.yandex-team.ru/runtime-cloud/RTC-SLA-Tentacles/dev/Drafts/monitoring-model-reorganization/ .

    # Shows if container is in Juggler children's list.
    monitoring_timestamp_age_present = UInt8Field()
    # Shows if container has some specific state in Juggler.
    monitoring_timestamp_age_visible = UInt8Field()
    # Shows if container answers on HTTP request.
    monitoring_timestamp_age_reachable = UInt8Field()
    # Shows if timestamp resource inside container is fresh enough.
    monitoring_timestamp_age_fresh_ts_resource = UInt8Field()

    # Binary columns to calculate Availability SLO.
    slo_availablilty_excluded = UInt8Field()
    slo_availablilty_available = UInt8Field()

    # Binary columns to calculate Redeployed on time SLO.
    slo_redeployment_excluded = UInt8Field()
    slo_redeployment_fresh_ts_resource = UInt8Field()


class WalleHost(TentaclesModel):

    @classmethod
    def table_name(cls):
        return "walle_host"

    ts = DateTimeField()
    fqdn = StringField()

    tentacles_location = StringField()
    rtc_stage = Enum8Field(rtc_stage_enum)
    deploy_gencfg = UInt8Field()
    deploy_yp_lite = UInt8Field()

    project = StringField()
    location_short_datacenter_name = StringField()
    operation_state = NullableField(Enum8Field(walle_operation_state_enum))
    state = StringField()
    status = StringField()
    health_status = NullableField(StringField())
    health_fail_reasons = ArrayField(Enum8Field(walle_health_fail_enum))

    scenario_id = UInt64Field()
    ticket = StringField()


class Iteration(TentaclesModel):

    @classmethod
    def table_name(cls):
        return "iterations"

    ts = DateTimeField()

    juggler_data_freshness = DateTimeField()
    hq_data_freshness = DateTimeField()
    walle_data_freshness = DateTimeField()
    yp_lite_allocation_data_freshness = DateTimeField()
    gencfg_data_freshness = DateTimeField()


nanny_current_state_values = ["OTHER", "OFFLINE", "ONLINE", "UPDATING", "SHUTTING_DOWN", "PREPARING"]
nanny_current_state_enum = DefaultEnum("nanny_current_state_enum", nanny_current_state_values)

nanny_reallocation_state_status_values = ["OTHER", "IDLE", "IN_PROGRESS", "ERROR"]
nanny_reallocation_state_status_enum = DefaultEnum("reallocation_state_status_enum",
                                                   nanny_reallocation_state_status_values)


class NannyState(TentaclesModel):

    history_depth = 4 * 24 * 60 * 60

    @classmethod
    def table_name(cls):
        return "nanny_state"

    ts = DateTimeField()
    deploy_gencfg = NullableField(UInt8Field())
    deploy_yp_lite = NullableField(UInt8Field())

    nanny_service_name = Enum8Field(nanny_service_name_enum)
    current_state = Enum8Field(nanny_current_state_enum)
    reallocation_id = StringField()
    reallocation_state_status = Enum8Field(nanny_reallocation_state_status_enum)
    reallocation_taskgroup_id = NullableField(StringField())
    latest_snapshot_id = NullableField(StringField())
    latest_snapshot_taskgroup_id = NullableField(StringField())


yp_cluster_name_values = [
    "OTHER",
    "SAS",
    "MAN",
    "VLA",
    "MYT",
    "IVA",
    "XDC",
    "MAN_PRE",
    "TEST_SAS"
]
yp_cluster_name_enum = DefaultEnum("yp_cluster_name_enum", yp_cluster_name_values)

yp_pod_disk_req_enum = DefaultEnum("yp_pod_disk_req_enum", ["hdd", "ssd"])


class YpLitePods(TentaclesModel):

    history_depth = 4 * 24 * 60 * 60

    @classmethod
    def table_name(cls):
        return "yp_lite_pods"

    ts = DateTimeField()
    nanny_service_name = Enum8Field(nanny_service_name_enum)
    yp_cluster = Enum8Field(yp_cluster_name_enum)

    yp_node_id = NullableField(StringField())

    # NOTE(rocco66): node was transfered to other segment, see TENTACLES-375
    yp_node_matches_node_filter = UInt8Field()

    yp_node_has_hdd = NullableField(UInt8Field())
    yp_node_has_ssd = NullableField(UInt8Field())
    yp_node_hfsm_state = NullableField(StringField())

    yp_pod_id = NullableField(StringField())
    yp_pod_fqdn = NullableField(StringField())
    yp_pod_disk_req = NullableField(Enum8Field(yp_pod_disk_req_enum))
    nanny_instance_pod_id = NullableField(StringField())
    scheduling_hint_node_id = NullableField(StringField())


def get_last_timestamp(return_timestamp=False):
    column = "max(ts) as max_ts"
    if return_timestamp:
        column = f"toUnixTimestamp({column})"
    return f"(SELECT {column} from {Iteration.table_name()})"


def to_clickhouse_datetime(input_ts):
    return datetime.datetime.utcfromtimestamp(int(input_ts)).strftime(_CLICKHOUSE_DATETIME_FORMAT)


ROTATING_TABLES = [Tentacle, WalleHost, Iteration, YpLitePods, NannyState]
