import json
import re
from infra.rtc_sla_tentacles.backend.lib.clickhouse.database import TABLE_OBJECT_MAPPING

from infi.clickhouse_orm.engines import MergeTree


def get_database_columns(client, table_name, with_types=False):
    table_data = json.loads(client.raw(f"DESCRIBE TABLE {table_name} FORMAT JSONCompact"))
    fields = [(i[0], i[1]) if with_types else i[0] for i in table_data["data"]]
    return fields


def get_configured_columns(table_name, with_types=False):
    model = TABLE_OBJECT_MAPPING[table_name]
    return [
        (name, re.match('\\w+(\\(.*\\))?', type_.get_sql().replace(" ,", ", "))[0]) if with_types else name
        for name, type_ in model._writable_fields.items()
    ]


def create_table(client, table_name, replica_name=None, replica_table_path=None, partition_key=None):
    # NOTE(rocco66): do not call this method from runtime code, CLI utils usage only
    table = TABLE_OBJECT_MAPPING[table_name]

    table.engine = MergeTree(
        date_col=table.date_col,
        order_by=table.order_by,
        replica_name=replica_name,
        replica_table_path=replica_table_path,
        partition_key=partition_key,
    )
    client.create_table(table)


def get_diff_between_tables(client, table_name):
    remote_columns = dict(get_database_columns(client, table_name, with_types=True))
    local_columns = dict(get_configured_columns(table_name, with_types=True))

    removed_columns = set(item for item in remote_columns.items() if item[0] not in local_columns)
    appended_columns = set()
    modified_columns = set()

    for item in local_columns.items():
        if item[0] not in remote_columns:
            appended_columns.add(item)
        elif item[1] != remote_columns[item[0]]:
            modified_columns.add((item[0], remote_columns[item[0]], item[1]))
    return appended_columns, removed_columns, modified_columns
