# encoding: utf-8
import copy
import os

from infra.rtc_sla_tentacles.backend.lib.config.exceptions import ConfigGetOptionError


class BaseConfig(object):
    """
        Abstract config holder class.
    """
    def __init__(self,
                 full_config,
                 secrets_config=None):
        """
            Receive settings, validate their structure and
            types, then replace secrets stubs with actual
            values.

        """
        self._db_entity_name_re = r"^[_\w\d]+$"
        self._secret_name_re = r"^[_A-Z\\d]+$"
        self._valid_datacenter_names = ["sas", "man", "msk", "vla", "test-run-no-dc"]
        self._valid_env_names = ["testing", "production"]

        self._secrets_config = secrets_config

        self._config = self._validate_and_return_part_of_full_config(full_config)
        self._resolve_secret_values_from_stubs()

    def to_dict(self):
        config_copy = copy.deepcopy(self._config)
        return config_copy

    def get_option(self, *path_to_option, mandatory=True):
        current_path = self._config
        human_readable_current_path = ["config_root"]

        for part in path_to_option:
            if part in current_path:
                human_readable_current_path.append(part)
                current_path = current_path[part]
            else:
                if not mandatory:
                    return None
                raise ConfigGetOptionError("{} is not found in {}".format(part, "/".join(human_readable_current_path)))
        return copy.deepcopy(current_path)

    def _resolve_secret_values_from_stubs(self):
        """
            Searches for secrets names' values, adds them to 'self._config'
            elements as another key-value element. For example,
            {
                "username": "foo",
                "password_secret_name": "SECRET_NAME"
            }

            in 'self._config' tells this function to add element with real
            password:
            {
                "username: "foo",
                "password_secret_name": "SECRET_NAME",
                "password": "bar"
            }

            Keys' names are just a convention.
            If secret's value can not be found in ENV or secret
            file - SecretsConfig object will raise ConfigValidationError.
        """
        raise NotImplementedError

    def _validate_and_return_part_of_full_config(self, full_config):
        """
            Validates part of a given 'full_config' using jsonschema, and,
            optionally, performs sanity checks on some config values.
            Returns validated part of a 'full_config' that holds only needed
            data in this specific class.
        """
        raise NotImplementedError

    @staticmethod
    def _check_file_exists_and_readable(path_to_file):
        """
            Check that 'path_to_file' is a file and can be read.
        """
        file_exists_and_readable = (
            os.path.isfile(path_to_file) and
            os.access(path_to_file, os.R_OK)
        )
        return file_exists_and_readable
