# encoding: utf-8
import copy

from jsonschema import validate

from infra.rtc_sla_tentacles.backend.lib.config.base_config import BaseConfig
from infra.rtc_sla_tentacles.backend.lib.config.exceptions import ConfigValidationError


class ClickhouseConfig(BaseConfig):
    """
        Holds Clickhouse connections and credentials.
    """
    def _resolve_secret_values_from_stubs(self):
        credentials = self._config["credentials"]
        for credential_name, credential in credentials.items():
            password_secret_name = credential["password_secret_name"]
            if not password_secret_name:  # Allow empty passwords.
                credentials[credential_name]["password"] = None
                continue
            password = self._secrets_config.get_secret(password_secret_name)
            credentials[credential_name]["password"] = password

    def _validate_and_return_part_of_full_config(self, full_config):
        clickhouse_config_schema = {
            "type": "object",
            "properties": {
                "storage": {
                    "type": "object",
                    "properties": {
                        "clickhouse": {
                            "type": "object",
                            "properties": {

                                "connection": {
                                    "type": "object",
                                    "additionalProperties": {"$ref": "#/definitions/clickhouse_connection_item"},
                                    "required": ["default"],
                                },

                                "credentials": {
                                    "type": "object",
                                    "additionalProperties": {"$ref": "#/definitions/clickhouse_credentials_item"},
                                    "required": ["default"],
                                }
                            },
                            "required": ["connection", "credentials"],
                        },
                    },
                    "required": ["clickhouse"],
                },
            },
            "required": ["storage"],

            "definitions": {

                "db_entity_name": {
                    "type": "string",
                    "pattern": self._db_entity_name_re,
                },

                "secret_name": {
                    "type": "string",
                    "pattern": self._secret_name_re,
                },

                "clickhouse_single_host": {
                    "type": "object",
                    "properties": {

                        "fqdn": {"type": "string", "format": "hostname"},
                        "port": {"type": "integer", "minimum": 1024, "maximum": 65535},
                        "datacenter_name": {
                            "anyOf": [
                                {"type": "string", "enum": self._valid_datacenter_names},
                                {"type": "null"},
                            ],
                        },
                    },
                    "required": ["fqdn", "port", "datacenter_name"],
                },

                "clickhouse_hosts_list": {
                    "type": "array",
                    "items": [{"$ref": "#/definitions/clickhouse_single_host"}],
                    "minItems": 1,
                },

                "clickhouse_connection_item": {
                    "type": "object",
                    "properties": {
                        "hosts": {"$ref": "#/definitions/clickhouse_hosts_list"},
                        "ssl_ca_cert_path": {
                            "anyOf": [
                                {"type": "string", "minLength": 1},
                                {"type": "null"}
                            ]
                        },
                        "database_name": {"$ref": "#/definitions/db_entity_name"},
                        "connect_timeout": {"type": "integer", "minimum": 1, "maximum": 100},
                        "read_timeout": {"type": "integer", "minimum": 1, "maximum": 100},
                        "native_port": {"type": "integer", "minimum": 1024, "maximum": 65535}
                    },
                    "required": [
                        "hosts",
                        "ssl_ca_cert_path",
                        "database_name",
                        "connect_timeout",
                        "read_timeout",
                        "native_port"
                    ],
                },

                "clickhouse_credentials_item": {
                    "type": "object",
                    "properties": {
                        "username": {"$ref": "#/definitions/db_entity_name"},
                        "password_secret_name": {
                            "anyOf": [
                                {"$ref": "#/definitions/secret_name"},
                                {"type": "null"},
                            ],
                        },
                    },
                    "required": ["username", "password_secret_name"],
                }
            },
        }
        validate(full_config, clickhouse_config_schema)

        # Check that 'ssl_ca_cert_path' file in all 'connections' exists and readable.
        connections = full_config["storage"]["clickhouse"]["connection"]
        for connection_name, connection_data in connections.items():
            # We set connection(s) with nearest DC first so the user
            # would try 'em first.
            connection_data['hosts'].sort(
                key=lambda h: h['datacenter_name'] != full_config['misc']['my_datacenter_name']
            )
            ssl_ca_cert_path = connection_data["ssl_ca_cert_path"]
            if not ssl_ca_cert_path:  # Skip empty 'ssl_ca_cert_path'.
                continue
            file_exists_and_readable = self._check_file_exists_and_readable(ssl_ca_cert_path)
            if not file_exists_and_readable:
                raise ConfigValidationError(
                    f"'storage' : 'clickhouse' : '{connection_name}' : 'ssl_ca_cert_path' -- "
                    f"{ssl_ca_cert_path!r} is not a file that can be read.")

        clickhouse_config = copy.deepcopy(full_config["storage"]["clickhouse"])
        return clickhouse_config
