# encoding: utf-8
import ssl
import itertools

from urllib.parse import quote_plus as quote

from infra.rtc_sla_tentacles.backend.lib.config.exceptions import ConfigGetOptionError
from infra.rtc_sla_tentacles.backend.lib.config.clickhouse_config import ClickhouseConfig
from infra.rtc_sla_tentacles.backend.lib.config.incidents import IncidentsConfig
from infra.rtc_sla_tentacles.backend.lib.config.harvesters_config import HarvestersConfig
from infra.rtc_sla_tentacles.backend.lib.config.logging_config import LoggingConfig
from infra.rtc_sla_tentacles.backend.lib.config.misc_config import MiscConfig
from infra.rtc_sla_tentacles.backend.lib.config.mongo_config import MongoConfig
from infra.rtc_sla_tentacles.backend.lib.config.nanny_config import NannyConfig
from infra.rtc_sla_tentacles.backend.lib.config.raw_config_storage import RawConfigStorage
from infra.rtc_sla_tentacles.backend.lib.config.secrets_config import SecretsConfig
from infra.rtc_sla_tentacles.backend.lib.config.tentacles_groups_config import TentaclesGroupsConfig
from infra.rtc_sla_tentacles.backend.lib.config.worker_config import WorkerConfig
from infra.rtc_sla_tentacles.backend.lib.config.api_config import ApiConfig
from infra.rtc_sla_tentacles.backend.lib.config.yp_config import YpConfig
from infra.rtc_sla_tentacles.backend.lib.config.yp_lite_pods_manager_config import YpLitePodsManagerConfig
from infra.rtc_sla_tentacles.backend.lib.config import utils as config_utils


class ConfigInterface(object):
    """
        Config interface.
    """
    def __init__(self, raw_config_storage=None, full_config=None, configure_logging=True, sentry_sender=None):
        if not full_config:
            if raw_config_storage:
                self._raw_config_storage = raw_config_storage
            else:
                self._raw_config_storage = RawConfigStorage()
            full_config = self._raw_config_storage.to_dict()

        self.secrets_config = SecretsConfig(full_config=full_config)
        self.mongo_config = MongoConfig(full_config=full_config, secrets_config=self.secrets_config)
        self.clickhouse_config = ClickhouseConfig(full_config=full_config, secrets_config=self.secrets_config)
        self.misc_config = MiscConfig(full_config=full_config)
        self.worker_config = WorkerConfig(full_config=full_config)
        self.harvesters_config = HarvestersConfig(full_config=full_config)
        if configure_logging:
            self.logging_config = LoggingConfig(full_config=full_config)
        self.api_config = ApiConfig(full_config=full_config)
        self.nanny_config = NannyConfig(full_config=full_config, secrets_config=self.secrets_config)
        self.yp_config = YpConfig(full_config=full_config, secrets_config=self.secrets_config)
        self.yp_lite_pods_manager_config = YpLitePodsManagerConfig(full_config=full_config)
        self.tentacles_groups_config = TentaclesGroupsConfig(full_config=full_config)
        self.incidents_config = IncidentsConfig(full_config=full_config)

        if sentry_sender:
            sentry_sender.environment = self.get_env_name()
        self.sentry_sender = sentry_sender

    def get_env_name(self):
        """
            Returns `env_name` from `misc` config.
        """
        return self.misc_config.get_option("env_name")

    def get_my_datacenter_name(self):
        """
            Returns `my_datacenter_name` from `misc` config.
        """
        return self.misc_config.get_option("my_datacenter_name")

    def get_worker_config(self):
        return self.worker_config.get_option()

    def get_mongo_url_dict(self):
        return self.get_harvesters_results_storage_config()["db_url"]

    def get_incidents_config(self):
        """
           Returns dict with Mongo connection strings and other options for Incidents.
        """
        return {
            "allocation_zones": self.get_allocation_zones(),
            "assignees": self.incidents_config.get_option("assignees"),
            "incidents_collection_name": self.incidents_config.get_option("incidents_collection_name"),
        }

    def get_harvesters_results_storage_config(self):
        """
            Returns dict with Mongo connection strings and other options for Harvesters.
        """
        results_storage_config = {}
        mongodb_connection_name = self.harvesters_config.get_option("results_storage", "mongodb", "connection")
        mongodb_credentials_name = self.harvesters_config.get_option("results_storage", "mongodb", "credentials")
        mongodb_locks_collection_name = self.harvesters_config.get_option(
            "results_storage",
            "mongodb",
            "locks_collection_name",
        )

        try:
            pymongo_additional_params = self.harvesters_config.get_option(
                "results_storage",
                "mongodb",
                "pymongo_additional_params",
            )
        except ConfigGetOptionError:
            pymongo_additional_params = {}

        mongodb_collection_name = None

        results_storage_config["db_url"] = self._generate_mongodb_client_parameters(
            mongodb_connection_name,
            mongodb_credentials_name,
            mongodb_collection_name,
            **pymongo_additional_params
        )
        results_storage_config["locks_collection_name"] = mongodb_locks_collection_name

        return results_storage_config

    def get_configured_harvesters(self):
        """
            Returns list of configured harvesters.
        """
        return list(self.harvesters_config.get_option("harvesters").keys())

    def get_harvester_config(self, name):
        """
            Returns dict with specified harvester config.
        """
        return self.harvesters_config.get_option("harvesters", name)

    def get_api_config(self):
        api_config = {}
        api_config["port"] = self.api_config.get_option("server_config", "port")
        api_config["listen_queue"] = self.api_config.get_option("server_config", "listen_queue")
        api_config["processes"] = self.api_config.get_option("server_config", "processes")
        api_config["threads"] = self.api_config.get_option("server_config", "threads")
        api_config["logfile"] = self.api_config.get_option("server_config", "logfile")
        api_config["log_maxsize"] = self.api_config.get_option("server_config", "log_maxsize", mandatory=False)
        clickhouse_connection = self.api_config.get_option("server_config", "clickhouse", "connection")
        clickhouse_credentials = self.api_config.get_option("server_config", "clickhouse", "credentials")
        storage_params = self._generate_clickhouse_client_parameters(clickhouse_connection, clickhouse_credentials)
        api_config["clickhouse_parameters"] = storage_params
        return api_config

    def get_clickhouse_nodes(self):
        hosts = self.get_api_config()["clickhouse_parameters"]["hosts"]
        return list(itertools.chain.from_iterable([hosts["in_my_dc"], hosts["in_other_dcs"]]))

    def get_api_gui_config(self):
        return self.api_config.get_option("gui_config")

    def get_nanny_rest_client_config(self) -> dict:
        config = dict()
        config["token"] = self.nanny_config.get_option("nanny_oauth_token")
        config["url"] = "https://nanny.yandex-team.ru/"
        config["timeout"] = 30
        return config

    def get_nanny_rpc_client_config(self, rpc_url_name: str) -> dict:
        rpc_urls_map = {
            "yp_lite_pod_sets_url": "https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/pod-sets/",
            "yp_lite_pod_reallocation_url": "https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/pod-reallocation/"
        }
        config = dict()
        config["oauth_token"] = self.nanny_config.get_option("nanny_oauth_token")
        config["rpc_url"] = rpc_urls_map[rpc_url_name]
        config["request_timeout"] = 30
        return config

    def get_allocation_zones(self, yp_daemonsets_only=False) -> dict:
        result = self.tentacles_groups_config.to_dict()
        if yp_daemonsets_only:
            return {
                zone_id: zone_config for zone_id, zone_config in result.items()
                if config_utils.is_daemonset_location(zone_id)
            }
        else:
            return result

    def get_allocation_zone_config(self, allocation_zone_id) -> dict:
        return self.tentacles_groups_config.to_dict().get(allocation_zone_id)

    def get_tentacles_groups_with_availability_configured(self):
        return [
            group_name
            for group_name, group_settings in self.tentacles_groups_config.to_dict().items()
            if group_settings.get("availability_settings")
        ]

    def get_tentacles_groups_with_reallocation(self):
        return [
            allocation_zone_id
            for allocation_zone_id in self.get_allocation_zones()
            if self.get_tentacles_group_reallocation_settings(allocation_zone_id)
        ]

    def get_tentacles_group_availability_settings(self, group_name):
        return self.tentacles_groups_config.get_option(group_name, "availability_settings")

    def get_tentacles_group_reallocation_settings(self, group_name):
        return self.tentacles_groups_config.get_option(group_name, "reallocation_settings")

    def get_tentacles_group_redeployment_settings(self, group_name):
        return self.tentacles_groups_config.get_option(group_name, "redeployment_settings")

    def get_yp_client_config(self, cluster: str) -> dict:
        address = self.yp_config.get_option("masters", cluster)
        config = self.yp_config.get_option("client_config_kwargs", mandatory=False) or {}
        config["token"] = self.yp_config.get_option("yp_token")
        return {
            "address": address,
            "config": config
        }

    def get_yp_lite_pods_manager_config(self) -> dict:
        config = dict()
        connection = self.yp_lite_pods_manager_config.get_option("clickhouse", "connection")
        credentials = self.yp_lite_pods_manager_config.get_option("clickhouse", "credentials")
        config["clickhouse_client_parameters"] = self._generate_clickhouse_client_parameters(connection, credentials)
        config["new_pods_allocation_batch_size"] = self.yp_lite_pods_manager_config.get_option(
            "new_pods_allocation_batch_size")
        config["pods_without_nodes_removal_batch_size"] = self.yp_lite_pods_manager_config.get_option(
            "pods_without_nodes_removal_batch_size")
        return config

    def _generate_clickhouse_client_parameters(self, connection_name, credentials_name):
        """
            Generates parameters for 'clickhouse_driver' client
            from given 'connection_name' and 'credentials_name' pair.

            Returns dict like
            {
                "hosts": {
                    "in_my_dc": [
                        {
                            "fqdn": "server2.example.com",
                            "port": 9440,
                            "datacenter_name": "man"
                        }
                    ],
                    "in_other_dcs": [
                        {
                            "fqdn": "server1.example.com",
                            "port": 9440,
                            "datacenter_name": "sas"
                        },
                        {
                            "fqdn": "server3.example.com",
                            "port": 9440,
                            "datacenter_name": "vla"
                        }
                    ]
                },
                "user": "some_username",
                "password": "some_password",
                "database": "some_database_name,
                "verify: True,
                "ca_certs": "/path_to_crt_file",
            }

            `hosts` are sorted from nearest to most distant from
            current DC.
        """
        connection = self.clickhouse_config.get_option("connection", connection_name)
        credentials = self.clickhouse_config.get_option("credentials", credentials_name)

        my_dc = self.get_my_datacenter_name() or "test-run-no-dc"
        # noinspection PyProtectedMember
        possible_dcs = self.misc_config._valid_datacenter_names.copy()

        hosts_lists = {}

        def _get_hosts_in_dc(_dc):
            _hosts_in_dc = [
                _host
                for _host in connection["hosts"]
                if _host["datacenter_name"] == _dc
            ]
            return _hosts_in_dc

        hosts_lists["in_my_dc"] = _get_hosts_in_dc(my_dc)
        hosts_lists["in_other_dcs"] = []
        possible_dcs.remove(my_dc)
        for dc in possible_dcs:
            hosts_in_dc = _get_hosts_in_dc(dc)
            hosts_lists["in_other_dcs"].extend(hosts_in_dc)

        password = credentials["password"]
        verify = bool(connection["ssl_ca_cert_path"])

        parameters = {
            "hosts": hosts_lists,
            "user": quote(credentials["username"]),
            "password": password,
            "database": connection["database_name"],
            "verify": verify,
            "ca_certs": connection["ssl_ca_cert_path"],
            "connect_timeout": connection["connect_timeout"],
            "read_timeout": connection["read_timeout"],
            "native_port": connection["native_port"]
        }

        return parameters

    def _generate_mongodb_client_parameters(self,
                                            connection_name,
                                            credentials_name,
                                            collection_name=None,
                                            **pymongo_additional_params):
        """
            Generates MongoClient parameters from given 'connection_name'
            and 'credentials_name' pair.
            Returns dict like:
            {
                "host": ("mongodb://{user}:{pw}@{hosts_line}/{db_name}.{collection_name}"
                         "?replicaSet={rs}&authSource={auth_src}"),
                "ssl_ca_certs": "/path_to_crt_file",
                "ssl_cert_reqs": ssl.CERT_REQUIRED, # or 'ssl.CERT_NONE' if 'ssl_ca_certs' is 'None'
                "additional_param1_name": "additional_param1_value" # parameters from 'pymongo_additional_params'
            }
        """
        connection = self.mongo_config.get_option("connection", connection_name)
        credentials = self.mongo_config.get_option("credentials", credentials_name)

        fqdn_port_pairs = []
        for host in connection["hosts"]:
            fqdn_port = ":".join([
                host["fqdn"],
                str(host["port"])
            ])
            fqdn_port_pairs.append(fqdn_port)

        hosts_line = ",".join(fqdn_port_pairs)

        username = quote(credentials["username"])
        password = credentials["password"]
        if password:
            password_quoted = quote(credentials["password"])
            username_password = f"{username}:{password_quoted}"
        else:
            username_password = username

        database_name = connection["database_name"]
        if collection_name:
            database_collection = f"{database_name}.{collection_name}"
        else:
            database_collection = database_name

        additional_url_params_list = []
        if connection["replicaset"]:
            replicaset_item = f"replicaSet={connection['replicaset']}"
            additional_url_params_list.append(replicaset_item)
        if credentials["auth_source"]:
            replicaset_item = f"authSource={credentials['auth_source']}"
            additional_url_params_list.append(replicaset_item)

        if additional_url_params_list:
            additional_url_params = "&".join(additional_url_params_list)
            url = f"mongodb://{username_password}@{hosts_line}/{database_collection}?{additional_url_params}"
        else:
            url = f"mongodb://{username_password}@{hosts_line}/{database_collection}"

        parameters = {
            "host": url,
            "ssl_ca_certs": connection["ssl_ca_cert_path"],
            "ssl_cert_reqs": ssl.CERT_REQUIRED if connection["ssl_ca_cert_path"] else ssl.CERT_NONE
        }
        if pymongo_additional_params:
            for param_name, param_value in pymongo_additional_params.items():
                parameters[param_name] = param_value
        return parameters

    def get_url(self):
        return self.misc_config.get_option("url")
