# encoding: utf-8
import copy

from jsonschema import validate

from infra.rtc_sla_tentacles.backend.lib.config.base_config import BaseConfig
from infra.rtc_sla_tentacles.backend.lib.config.exceptions import ConfigValidationError


class MongoConfig(BaseConfig):
    """
        Holds MongoDb connections and credentials.
    """
    def _resolve_secret_values_from_stubs(self):
        credentials = self._config["credentials"]
        for credential_name, credential in credentials.items():
            password_secret_name = credential["password_secret_name"]
            if not password_secret_name:  # Allow empty passwords.
                credentials[credential_name]["password"] = None
                continue
            password = self._secrets_config.get_secret(password_secret_name)
            credentials[credential_name]["password"] = password

    def _validate_and_return_part_of_full_config(self, full_config):
        mongo_config_schema = {
            "type": "object",
            "properties": {
                "storage": {
                    "type": "object",
                    "properties": {
                        "mongodb": {
                            "type": "object",
                            "properties": {

                                "connection": {
                                    "type": "object",
                                    "additionalProperties": {"$ref": "#/definitions/mongo_connection_item"},
                                    "required": ["default"],
                                },

                                "credentials": {
                                    "type": "object",
                                    "additionalProperties": {"$ref": "#/definitions/mongo_credentials_item"},
                                    "required": ["default"],
                                }
                            },
                            "required": ["connection", "credentials"],
                        },
                    },
                    "required": ["mongodb"],
                },
            },
            "required": ["storage"],

            "definitions": {

                "db_entity_name": {
                    "type": "string",
                    "pattern": self._db_entity_name_re,
                },

                "secret_name": {
                    "type": "string",
                    "pattern": self._secret_name_re,
                },

                "mongo_single_host": {
                    "type": "object",
                    "properties": {
                        "fqdn": {"type": "string", "format": "hostname"},
                        "port": {"type": "integer", "minimum": 1024, "maximum": 65535},
                    },
                    "required": ["fqdn", "port"],
                },

                "mongo_hosts_list": {
                    "type": "array",
                    "items": [{"$ref": "#/definitions/mongo_single_host"}],
                    "minItems": 1,
                },

                "mongo_connection_item": {
                    "type": "object",
                    "properties": {
                        "hosts": {"$ref": "#/definitions/mongo_hosts_list"},
                        "replicaset": {
                            "anyOf": [
                                {"$ref": "#/definitions/db_entity_name"},
                                {"type": "null"},
                            ],
                        },
                        "ssl_ca_cert_path": {
                            "anyOf": [
                                {"type": "string", "minLength": 1},
                                {"type": "null"},
                            ],
                        },
                        "database_name": {"$ref": "#/definitions/db_entity_name"},
                    },
                    "required": ["hosts", "replicaset", "ssl_ca_cert_path", "database_name"],
                },

                "mongo_credentials_item": {
                    "type": "object",
                    "properties": {
                        "username": {"$ref": "#/definitions/db_entity_name"},
                        "password_secret_name": {
                            "anyOf": [
                                {"$ref": "#/definitions/secret_name"},
                                {"type": "null"},
                            ],
                        },
                        "auth_source": {
                            "anyOf": [
                                {"$ref": "#/definitions/db_entity_name"},
                                {"type": "null"},
                            ],
                        },
                    },
                    "required": ["username", "password_secret_name", "auth_source"],
                },
            },
        }
        validate(full_config, mongo_config_schema)

        # Check that 'ssl_ca_cert_path' file in all 'connections' exists and readable.
        connections = full_config["storage"]["mongodb"]["connection"]
        for connection_name, connection_data in connections.items():
            ssl_ca_cert_path = connection_data["ssl_ca_cert_path"]
            if not ssl_ca_cert_path:
                continue
            file_exists_and_readable = self._check_file_exists_and_readable(ssl_ca_cert_path)
            if not file_exists_and_readable:
                raise ConfigValidationError(
                    f"'storage' : 'mongodb': 'connections' : '{connection_name}' : 'ssl_ca_cert_path' "
                    f"{ssl_ca_cert_path!r} is not a file that can be read."
                )

        mongo_config = copy.deepcopy(full_config["storage"]["mongodb"])
        return mongo_config
