# encoding: utf-8
import argparse
import copy
import os

from infra.rtc_sla_tentacles.backend.lib.config.exceptions import ConfigValidationError
from infra.rtc_sla_tentacles.backend.lib.util import modify_value_in_dict_by_path_or_raise,\
    read_yaml_file


class RawConfigStorage(object):
    """
        Object that reads and holds static application config
        options.
    """
    def __init__(self,
                 cli_args=None,
                 file_cfg=None):
        # Read configuration from args provided,
        # or from CLI and config file provided in CLI.
        if isinstance(cli_args, dict) and \
           isinstance(file_cfg, dict):
            self._cli_args = cli_args
            self._file_cfg = file_cfg
        else:
            self._cli_args = self._read_cli_arguments()
            _config_file_path = self._cli_args["CONFIG_FILE"]
            self._file_cfg = read_yaml_file(_config_file_path)

        self._config = self._merge_env_and_cli_to_file_config(self._file_cfg, self._cli_args)

    def to_dict(self):
        config = copy.deepcopy(self._config)
        return config

    @staticmethod
    def _merge_env_and_cli_to_file_config(file_cfg, cli_args):
        try:
            # Merge 'SECRETS_DIRECTORY' from CLI.
            if cli_args["SECRETS_DIRECTORY"]:
                modify_value_in_dict_by_path_or_raise(file_cfg,
                                                      "secrets_directory",
                                                      cli_args["SECRETS_DIRECTORY"],
                                                      "secrets")
            # Merge 'MY_DATACENTER_NAME' from CLI or ENV.
            my_datacenter_name = cli_args["MY_DATACENTER_NAME"]
            if not my_datacenter_name:
                my_datacenter_name = os.getenv("MY_DATACENTER_NAME")
            if not my_datacenter_name:
                my_datacenter_name = os.getenv('a_dc')
            if my_datacenter_name:
                modify_value_in_dict_by_path_or_raise(file_cfg,
                                                      "my_datacenter_name",
                                                      my_datacenter_name,
                                                      "misc")
            # Merge 'MONGO_DB_NAME' from CLI.
            if cli_args["MONGO_DB_NAME"]:
                modify_value_in_dict_by_path_or_raise(file_cfg,
                                                      "database_name",
                                                      cli_args["MONGO_DB_NAME"],
                                                      "storage",
                                                      "mongodb",
                                                      "connection",
                                                      "default")
            # Merge 'CH_DB_NAME' from CLI.
            if cli_args["CH_DB_NAME"]:
                modify_value_in_dict_by_path_or_raise(file_cfg,
                                                      "database_name",
                                                      cli_args["CH_DB_NAME"],
                                                      "storage",
                                                      "clickhouse",
                                                      "connection",
                                                      "default")
        except KeyError as _exc:
            raise ConfigValidationError(f"Cannot merge ENV and CLI options to file config: {_exc}")
        return file_cfg

    @staticmethod
    def _read_cli_arguments():
        parser = argparse.ArgumentParser(description="RTC SLA Tentacles backend")
        # Required config group.
        required_config_group = parser.add_argument_group()
        required_config_group.add_argument("-c",
                                           "--conf",
                                           required=True,
                                           dest="CONFIG_FILE",
                                           type=argparse.FileType("r"),
                                           help="Path to main configuration file.")
        # Optional group for databases' names.
        db_names_group = parser.add_argument_group(description=("Default databases names. "
                                                                "If set, these names will overwrite names in "
                                                                "configuration file."))
        db_names_group.add_argument("--mongodb-db-name",
                                    required=False,
                                    dest="MONGO_DB_NAME",
                                    help="Default MongoDb database name")
        db_names_group.add_argument("--clickhouse-db-name",
                                    required=False,
                                    dest="CH_DB_NAME",
                                    help="Default ClickHouse database name")
        # Optional group for secrets directory.
        secrets_group = parser.add_argument_group(description=(
            "Secrets are read in following order: "
            "  1. From files in \"secrets : secrets_directory\" parameter in configuration file. This parameter may be "
            "changed through \"--secrets-directory\" command line argument. "
            "  2. From ENV vars. Secret value that is set in ENV overwrites secret value that was read from config."))
        secrets_group.add_argument("--secrets-directory",
                                   required=False,
                                   dest="SECRETS_DIRECTORY",
                                   help="Path to directory with files with secrets.")
        # Optional group for configuring binary runtime environment.
        environment_group = parser.add_argument_group(description="Environment settings.")
        environment_group.add_argument("--my-datacenter-name",
                                       required=False,
                                       dest="MY_DATACENTER_NAME",
                                       help=("Name of the datacenter where this binary runs. "
                                             "Can also be set through ENV var called \"MY_DATACENTER_NAME\". "
                                             "Value set through command line argument have higher priority than ENV."))

        args = parser.parse_args()
        return vars(args)
