# encoding: utf-8
import copy
import os

from jsonschema import validate

from infra.rtc_sla_tentacles.backend.lib.config.base_config import BaseConfig
from infra.rtc_sla_tentacles.backend.lib.config.exceptions import ConfigValidationError


class SecretsConfig(BaseConfig):
    """
        Holds secrets settings. Does not holds secrets.
        Provides a method to read any secret from secret file
        or environment variable. ENV has more priority than file.
        Intended to be used as a secret getter in other Config
        objects.
    """
    def get_secret(self, secret_name):
        """
            Returns secret value of a given secret name.
            Reads ENV first, then files in secrets_directory (if present).
        """
        secret_value = os.getenv(secret_name)
        if secret_value is None and self._config["secrets_directory"]:
            secret_file = os.path.join(self._config["secrets_directory"],
                                       secret_name)
            secret_value = self._read_secret_file(secret_file)
        if secret_value is None:
            raise ConfigValidationError(f"Cannot find value for secret name {secret_name!r}.")
        return secret_value

    def _resolve_secret_values_from_stubs(self):
        # No secret values to resolve.
        pass

    @staticmethod
    def _read_secret_file(filename):
        try:
            with open(filename) as _f:
                contents = _f.readline().rstrip("\n")
                return contents
        except (IOError, OSError):
            return None

    def _validate_and_return_part_of_full_config(self, full_config):
        secrets_config_schema = {
            "type": "object",
            "properties": {
                "secrets": {
                    "type": "object",
                    "properties": {
                        "secrets_directory": {"$ref": "#/definitions/secrets_directory"},
                    },
                    "required": ["secrets_directory"],
                },
            },
            "required": ["secrets"],

            "definitions": {

                "secrets_directory": {
                    "anyOf": [
                        {"type": "string", "minLength": 1},
                        {"type": "null"},
                    ],
                },

            },
        }
        validate(full_config, secrets_config_schema)

        secrets_dir_used = full_config["secrets"]["secrets_directory"]
        secrets_dir_used_does_not_exists = (secrets_dir_used and not os.path.isdir(secrets_dir_used))
        if secrets_dir_used_does_not_exists:
            raise ConfigValidationError(f"Secrets directory {secrets_dir_used!r} provided is not a directory.")

        secrets_config = copy.deepcopy(full_config["secrets"])
        return secrets_config
