import copy

from jsonschema import validate

from infra.rtc_sla_tentacles.backend.lib.config.base_config import BaseConfig


# Defaults for availability and redeployment monitoring.
DEFAULT_MIN_PERCENT_OF_AVAILABLE_TENTACLES = 96
DEFAULT_MAX_PERCENT_OF_UNUSED_NODES = 5
DEFAULT_MAX_PERCENT_OF_SCHEDULING_ERRORS = 2
DEFAULT_MIN_PERCENT_OF_TENTACLES_WITH_FRESH_TS_RESOURCE = 96


class TentaclesGroupsConfig(BaseConfig):
    """
        Holds Tentacles groups config.
    """
    def _resolve_secret_values_from_stubs(self):
        # No secret values to resolve.
        pass

    def _validate_and_return_part_of_full_config(self, full_config):
        tentacles_groups_config_schema = {
            "type": "object",
            "properties": {
                "tentacles_groups": {
                    "type": ["object"],
                    "minProperties": 1,
                    "additionalProperties": {
                        "type": "object",
                        "properties": {
                            "availability_settings": {"$ref": "#/definitions/availability_settings"},
                            "reallocation_settings": {"$ref": "#/definitions/reallocation_settings"},
                            "redeployment_settings": {"$ref": "#/definitions/redeployment_settings"},
                        },
                    },
                },
            },
            "required": ["tentacles_groups"],


            "definitions": {

                "availability_settings": {
                    "type": ["object"],
                    "properties": {
                        "monitoring": {
                            "type": ["object", "null"],
                            "properties": {
                                "min_percent_of_available_tentacles": {
                                    "type": "integer",
                                    "minimum": 1,
                                    "maximum": 100,
                                    "title": "Percent of available tentacles, if less - launch CRIT",
                                },
                            },
                        },
                    }
                },

                "reallocation_settings": {
                    "type": ["object", "null"],
                    "properties": {
                        "degrade_params": {
                            "type": "object",
                            "properties": {
                                "max_unavailable_pods": {
                                    "type": "integer",
                                    "minimum": 1,
                                    "title": "Number of pods to reallocate at the same time",
                                },
                                "min_update_delay_seconds": {
                                    "type": "integer",
                                    "minimum": 1,
                                    "title": "Wait at least min_update_delay_seconds after pod reallocation",
                                }
                            },
                            "required": ["max_unavailable_pods", "min_update_delay_seconds"]
                        },
                        "cooldown_after_reallocation_min": {
                            "type": "integer",
                            "minimum": 1,
                            "maximum": 60,
                            "title": "Period of inactivity after reallocation ends, minutes",
                        },
                        "monitoring": {
                            "type": "object",
                            "properties": {
                                "reallocation_stalled_after_min": {
                                    "type": "integer",
                                    "minimum": 1,
                                    "title": "Reallocation is considered as stalled if lasts longer, minutes",
                                },
                                "last_reallocation_maximum_age_min": {
                                    "type": "integer",
                                    "minimum": 1,
                                    "title": "Last reallocation session must happen no more than N minutes ago",
                                },
                            },
                            "required": ["reallocation_stalled_after_min", "last_reallocation_maximum_age_min"],
                        },
                    },
                    "required": ["degrade_params", "cooldown_after_reallocation_min", "monitoring"],
                },

                "redeployment_settings": {
                    "type": ["object", "null"],
                    "properties": {
                        "resource_maker": {
                            "type": "string",
                            "minLength": 1,
                            "title": "ResourceMaker instance name",
                        },
                        "cooldown_after_redeployment_min": {
                            "type": "integer",
                            "minimum": 1,
                            "maximum": 60,
                            "title": "Period of inactivity after redeploy ends, minutes",
                        },
                        "monitoring": {
                            "type": "object",
                            "properties": {
                                "redeployment_stalled_after_min": {
                                    "type": "integer",
                                    "minimum": 1,
                                    "title": "Redeployment is considered as stalled if lasts longer, minutes",
                                },
                                "last_redeployment_maximum_age_min": {
                                    "type": "integer",
                                    "minimum": 1,
                                    "title": "Last redeployment session must happen no more than N minutes ago",
                                },
                                "min_percent_of_tentacles_with_fresh_resource": {
                                    "type": "integer",
                                    "minimum": 1,
                                    "maximum": 100,
                                    "title": "Percent of tentacles with fresh resource inside, if less - launch CRIT",
                                },
                            },
                            "required": [
                                "redeployment_stalled_after_min",
                                "last_redeployment_maximum_age_min",
                            ],
                        },
                        "update_nanny_instances_with_allocated_pods": {
                            "type": "boolean",
                            "title": (
                                "(Optional) Update Nanny instances list with allocated pods. Nanny service must be "
                                "tracked by 'yp_lite_pods_tracker' harvester.")
                        }
                    },
                    "required": ["resource_maker", "cooldown_after_redeployment_min"],
                },
            }
        }
        validate(full_config, tentacles_groups_config_schema)

        tentacles_groups_config = copy.deepcopy(full_config["tentacles_groups"])
        return tentacles_groups_config

    def get_availability_limit(self, allocation_zone_id):
        monitoring_config = self._config[allocation_zone_id].get("availability_settings", {}).get("monitoring", {})
        return (
            monitoring_config.get("min_percent_of_available_tentacles")
            or DEFAULT_MIN_PERCENT_OF_AVAILABLE_TENTACLES
        )

    def get_unused_nodes_limit(self, allocation_zone_id):
        monitoring_config = self._config[allocation_zone_id].get("unused_nodes_settings", {}).get("monitoring", {})
        return (
            monitoring_config.get("max_percent_of_unused_nodes")
            or DEFAULT_MAX_PERCENT_OF_UNUSED_NODES
        )

    def get_scheduling_errors_limit(self, allocation_zone_id):
        monitoring_config = self._config[allocation_zone_id].get("scheduling_settings", {}).get("monitoring", {})
        return (
            monitoring_config.get("max_percent_of_scheduling_errors")
            or DEFAULT_MAX_PERCENT_OF_SCHEDULING_ERRORS
        )

    def get_redeployment_limit(self, allocation_zone_id):
        monitoring_config = self._config[allocation_zone_id].get("redeployment_settings", {}).get("monitoring", {})
        return (
            monitoring_config.get("min_percent_of_tentacles_with_fresh_resource")
            or DEFAULT_MIN_PERCENT_OF_TENTACLES_WITH_FRESH_TS_RESOURCE
        )

    def get_flow_availability_limit(self, allocation_zone_id):
        monitoring_config = self._config[allocation_zone_id].get("availability_settings", {}).get("monitoring", {})
        return monitoring_config.get("flow_slo_limit")

    def get_flow_redeployment_limit(self, allocation_zone_id):
        monitoring_config = self._config[allocation_zone_id].get("redeployment_settings", {}).get("monitoring", {})
        return monitoring_config.get("flow_slo_limit")

    def is_reallocation_enable(self, allocation_zone_id):
        reallocation_settings = self._config[allocation_zone_id].get("reallocation_settings") or {}
        return bool(reallocation_settings.get("monitoring"))
