import traceback
import collections
import contextlib
import threading
import time

import flask

from infra.yasm import unistat
from infra.rtc_sla_tentacles.backend.lib.funccall_stats_server.memview import memview_handler


HarvesterStats = collections.namedtuple("HarvesterStats", ["time", "count", "errors"])


unistat_ = unistat.global_unistat

_worker_busy_metric = unistat_.create_float(
    "worker_busy_count", suffix="ammv", aggregation_type=unistat.AggregationType.LastValue, always_visible=True
)
_worker_free_metric = unistat_.create_float(
    "worker_free_count", suffix="ammv", aggregation_type=unistat.AggregationType.LastValue, always_visible=True
)

_mongo_requests_metric = unistat_.create_histogram("mongodb_requests", always_visible=True)
_clickhouse_requests_metric = unistat_.create_histogram("clickhouse_requests", always_visible=True)

_yp_requests_metric = unistat_.create_histogram("yp_requests", always_visible=True)
_nanny_requests_metric = unistat_.create_histogram("nanny_requests", always_visible=True)
_yp_lite_requests_metric = unistat_.create_histogram("yp_lite_requests", always_visible=True)
_juggler_requests_metric = unistat_.create_histogram("juggler_requests", always_visible=True)
_infra_requests_metric = unistat_.create_histogram("infra_requests", always_visible=True)
_walle_requests_metric = unistat_.create_histogram("walle_requests", always_visible=True)
_harvester_stats = {}


def mongo_timing(*args, **kwargs):
    return _add_timing(_mongo_requests_metric, *args, **kwargs)


def clickhouse_timing(*args, **kwargs):
    return _add_timing(_clickhouse_requests_metric, *args, **kwargs)


def yp_timing(*args, **kwargs):
    return _add_timing(_yp_requests_metric, *args, **kwargs)


def nanny_timing(*args, **kwargs):
    return _add_timing(_nanny_requests_metric, *args, **kwargs)


def yp_lite_timing(*args, **kwargs):
    return _add_timing(_yp_lite_requests_metric, *args, **kwargs)


def juggler_timing(*args, **kwargs):
    return _add_timing(_juggler_requests_metric, *args, **kwargs)


def infra_timing(*args, **kwargs):
    return _add_timing(_infra_requests_metric, *args, **kwargs)


def walle_timing(*args, **kwargs):
    return _add_timing(_walle_requests_metric, *args, **kwargs)


@contextlib.contextmanager
def _add_timing(metric):
    start_time = time.time()
    yield
    delta = time.time() - start_time
    metric.push(delta)


def add_error(harvester_type_str, exec_time):
    _add_harvester_stats(harvester_type_str, exec_time, is_error=True)


def add_ok(harvester_type_str, exec_time):
    _add_harvester_stats(harvester_type_str, exec_time, is_error=False)


def _add_harvester_stats(name, time_spent, is_error):
    _harvester_stats[name].count.push(1)
    _harvester_stats[name].time.push(time_spent)
    if is_error:
        _harvester_stats[name].errors.push(1)


def init_harvester_unistat(harvester_types):
    # for harvester_group in harvester_manager.get_harvester_groups():
    #     harvester_type = harvester_group.harvester_type
    for harvester_type in harvester_types:
        _harvester_stats[harvester_type] = HarvesterStats(
            unistat_.create_histogram(f"task_{harvester_type}_time", always_visible=True),
            unistat_.create_float(f"task_{harvester_type}_count", suffix="dmmm", always_visible=True),
            unistat_.create_float(f"task_{harvester_type}_errors", suffix="dmmm", always_visible=True),
        )


class FunccallStatsServer:
    """
    Account and report worker stats for yasm.
    Each task adds its timings and error flag to stats after call,
    and these stats will be reset each time when yasm retrieves data.
    """

    def __init__(self, name: str, port: int, thread_count):
        self.name = name
        self.port = port

        self._total_threads = thread_count
        self._free_harvester_lock = threading.Lock()
        self._free_harvester = thread_count

    def run_flask(self):
        app = self._get_flask_app()
        app.run(host="::", port=self.port, debug=False, load_dotenv=False)

    def _get_flask_app(self):
        """
        Thread that serves flask application with unistat handle for yasm
        """

        app = flask.Flask(self.name)

        @app.route('/ping', methods=['GET'])
        def ping():
            return "OK"

        @app.route("/unistat", methods=["GET"])
        def unistat_handler():
            real_free_threads = max(0, self._free_harvester)
            _worker_busy_metric.push(self._total_threads - real_free_threads)
            _worker_free_metric.push(real_free_threads)
            return unistat_.to_json(all_signals=True), 200

        @app.route("/memview", methods=["GET"])
        def memview():
            try:
                return memview_handler(flask.request.args.get('method'))
            except Exception:
                error = traceback.format_exc()
                print(error)
                return error

        return app

    def harvester_acquire(self):
        with self._free_harvester_lock:
            self._free_harvester -= 1

    def harvester_free(self):
        with self._free_harvester_lock:
            self._free_harvester += 1

    def get_free_harvesters(self):
        return self._free_harvester
