import contextlib
import logging
import time
import typing

from jsonschema import validate

from infra.rtc_sla_tentacles.backend.lib.config.interface import ConfigInterface
from infra.rtc_sla_tentacles.backend.lib.self_monitoring import juggler_self_monitoring
from infra.rtc_sla_tentacles.backend.lib.harvesters_snapshots.manager import HarvesterSnapshotManager
from infra.rtc_sla_tentacles.backend.lib.harvesters_snapshots.snapshots import HarvesterSnapshot, HarvesterSnapshotLabel
from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager import harvesters_snapshots_freshness_checks


class Harvester:
    run_on_all_workers = False
    secrets_map: typing.Dict[str, typing.Optional[str]] = {}
    harvester_type = ""
    _base_type_config_schema = {
        "type": "object",
        "properties": {
            "common_settings": {
                "type": "object",
                "properties": {
                    "chunk_size": {
                        "type": "integer",
                    },
                    "data_list_path": {
                        "type": ["string", "null"],
                    },
                    "update_interval_sec": {
                        "type": "integer",
                    },
                    "rotate_snapshots_older_than_sec": {
                        "type": "integer",
                    },
                },
                "required": ["chunk_size", "data_list_path", "update_interval_sec", "rotate_snapshots_older_than_sec"],
            },
            "common_parameters": {
                "type": ["object", "null"],
            },
            "arguments": {
                "type": "object",
                "minProperties": 1,
                "additionalProperties": {
                    "type": ["object", "null"],
                }
            }
        },
        "required": ['common_settings', 'common_parameters', 'arguments'],
    }
    _type_config_schema: typing.Optional[dict] = None
    _instance_config_schema: typing.Optional[dict] = None

    def __init__(
        self,
        name: str,
        arguments: dict,
        common_parameters,
        common_settings,
        snapshot_manager: HarvesterSnapshotManager,
        config_interface: ConfigInterface,
        several_harvesters: bool,
    ):
        self.name = name
        self.arguments = arguments or {}
        self.common_parameters = common_parameters
        self.common_settings = common_settings
        self._snapshot_manager = snapshot_manager
        self.config_interface = config_interface

        self.logger = logging.getLogger(f'harvesters.{self.harvester_type}.{self.name}')
        self.logger.debug('Created harvester instance')

        self._several_harvesters = several_harvesters
        self.juggler_sender = juggler_self_monitoring.JugglerSender(self.logger, self.config_interface)

    def run(self, ts: int):
        """
        Непосредственное выполнение полезной работы
        :param int ts: Время запуска
        """
        debug_info: typing.Dict[str, typing.Any] = {
            "extract_start_time": int(time.time()),
            "extract_end_time": None,
            "extract_exception": None,
            "transform_start_time": None,
            "transform_end_time": None,
            "transform_exception": None
        }

        self.logger.info(f"{self.__class__.__name__} harvester extract iteration, ts=%d", ts)
        try:
            raw_data = self.extract(ts)
        except Exception:
            self.logger.exception("harvester error")
            return
        debug_info["extract_end_time"] = int(time.time())

        self.logger.info(f"{self.__class__.__name__} harvester transform iteration, ts=%d", ts)
        debug_info["transform_start_time"] = int(time.time())
        try:
            meta, data = self.transform(ts, raw_data)
        except Exception:
            self.logger.exception("harvester error")
            return
        debug_info["transform_end_time"] = int(time.time())
        snapshot = self._create_snapshot(ts, debug_info, meta, data)
        try:
            with self.time_it("write result snapshot"):
                self._snapshot_manager.write_snapshot(snapshot)
        except Exception:
            self.logger.exception("uncaught write snapshot error")
        self._report_harvester_freshness()

        try:
            with self.time_it("snapshot rotation"):
                self._rotate_snapshots(ts=ts)
        except Exception:
            self.logger.exception("harvester rotate error")
            return

        self.logger.debug("run exit: ts=%d", ts)

    def _report_harvester_freshness(self):
        self.juggler_sender.ok(
            host=harvesters_snapshots_freshness_checks.make_juggler_host(self.config_interface.get_env_name()),
            service=harvesters_snapshots_freshness_checks.make_juggler_service(
                self.harvester_type, self.name, self._several_harvesters,
            )
        )

    def _create_snapshot_label(self, ts: int) -> HarvesterSnapshotLabel:
        return HarvesterSnapshotLabel(ts=ts,
                                      harvester_type=self.harvester_type,
                                      harvester_name=self.name,
                                      chunk_size=self.common_settings["chunk_size"],
                                      data_list_path=self.common_settings["data_list_path"])

    def _create_snapshot(self,
                         ts: int,
                         debug_info: typing.Dict,
                         meta: typing.Any,
                         data: typing.Any,
                         label: HarvesterSnapshotLabel = None) -> HarvesterSnapshot:
        if not label:
            label = self._create_snapshot_label(ts)
        return HarvesterSnapshot(label=label,
                                 debug_info=debug_info,
                                 meta=meta,
                                 data=data)

    @classmethod
    def _validate_type_config(cls, config):
        # We could invent some kind of schema merging from base class
        # to descendants, but there're too many implicit cases, than
        # would confuse and lead to mistakes, so one shouldn't do it.
        # Hence we validate base schema related to  base class and let
        # type-specific schema be a separate one from descendant class.
        if cls._base_type_config_schema is not None:
            validate(config, cls._base_type_config_schema)
        if cls._type_config_schema is not None:
            validate(config, cls._type_config_schema)
        if cls._instance_config_schema is not None:
            for instance in config['arguments'].values():
                validate(instance, cls._instance_config_schema)

    @classmethod
    def build_instances(cls, config_interface, snapshot_manager, type_config):
        cls._validate_type_config(type_config)
        for name, arguments in type_config['arguments'].items():
            yield cls(
                name,
                arguments,
                type_config['common_parameters'],
                type_config['common_settings'],
                snapshot_manager,
                config_interface,
                several_harvesters=len(type_config['arguments']) > 1

            )

    def get_interval(self) -> int:
        return self.common_settings["update_interval_sec"]

    def extract(self, ts: int) -> typing.Any:
        """
        Получение инфы из внешнего источника
        :return: Распаршенная выдача из внешнего источника
        """
        raise NotImplementedError

    def transform(self, ts: int, data) -> typing.Tuple[typing.Any, typing.Any]:
        """
        Приведение инфы из внешнего источника в компактный вид для сохранения в Монгу.
        :param ts: Время запуска
        :param data: Any data
        """
        return {}, data

    def _rotate_snapshots(self, ts: int) -> None:
        """
            Delete this harvester's snapshots, which are older than
            `rotate_snapshots_older_than_sec`. Keep one most fresh
            successful snapshot regardless of its age.
        """
        rotate_border = ts - self.common_settings["rotate_snapshots_older_than_sec"]
        self._snapshot_manager.clean_old_snapshots(self.harvester_type, self.name, rotate_border)
        self.logger.info("Snapshots rotated, border=%d" % rotate_border)

    @contextlib.contextmanager
    def time_it(self, section_name):
        start_time = time.time()
        yield
        delta = time.time() - start_time
        self.logger.info(f"'{section_name}' execution time {delta:.2f} sec")
