import typing
import json

from infra.rtc_sla_tentacles.backend.lib.clickhouse.client import ClickhouseClient
from infra.rtc_sla_tentacles.backend.lib.clickhouse import database
from infra.rtc_sla_tentacles.backend.lib.harvesters.base import Harvester


class ClickhouseOptimizer(Harvester):
    harvester_type = "clickhouse_optimizer"

    def _make_client(self):
        clickhouse_params = self.config_interface.get_api_config()["clickhouse_parameters"]
        self.database = clickhouse_params["database"]
        return ClickhouseClient(self.config_interface, leader_only=True)

    def extract(self, ts: int) -> typing.Any:
        client = self._make_client()
        result = {}
        for table_class in database.ROTATING_TABLES:
            table_name = table_class.table_name()
            query = f"""
            SELECT partition
            FROM
            (
                SELECT
                    partition,
                    COUNT() AS cnt,
                    MAX(modification_time) AS mod_time
                FROM system.parts
                WHERE (table = '{table_name}') AND (active = 1)
                GROUP BY
                    partition
                ORDER BY
                    cnt DESC,
                    mod_time DESC
            )
            WHERE ((now() - mod_time) / 60) > 15 AND (cnt > 1)
            LIMIT 1
            FORMAT JSONCompact
            """
            response = json.loads(client.raw(query))
            if response["data"]:
                result[table_name] = response["data"][0][0]
        return result

    def transform(self, ts: int, data) -> typing.Tuple[typing.Any, typing.Any]:
        if not data:
            return {"optimized": {}}, {}
        client = self._make_client()
        result = {}
        for table_name, partition in data.items():
            client.raw(f"OPTIMIZE TABLE {table_name} PARTITION {partition} FINAL")
            result[table_name] = 1
        return {"optimized": result}, {}
