import requests
import typing

import dataclasses

from infra.rtc_sla_tentacles.backend.lib.harvesters.base import Harvester
from infra.rtc_sla_tentacles.backend.lib.util import requests_retry_session
from infra.rtc_sla_tentacles.backend.lib.funccall_stats_server import server as stat_server


EVENTS_FILTER_FROM_HOURS = 24 * 7  # last week
EVENTS_FILTER_TO_HOURS = 24 * 7  # next week


@dataclasses.dataclass
class InfraEvent:
    id: int
    dcs: typing.List[str]
    start_time: int
    finish_time: int

    def to_link(self):
        return f"https://infra.yandex-team.ru/event/{self.id}"


class InfraEventsCacherError(Exception):
    """Can't fetch events fron infra.yandex-team.ru"""


class InfraEventsCacher(Harvester):
    harvester_type = "infra_events_cacher"

    def extract(self, ts: int) -> typing.Any:
        session = requests.session()
        session.headers["Content-Type"] = "application/json"
        session.headers["Accept"] = "application/json"
        session_with_retries = requests_retry_session(session=session)

        with stat_server.infra_timing():
            response = session_with_retries.get(url="https://infra-api.yandex-team.ru/v1/events",
                                                params={
                                                    "from": ts - (60 * 60 * EVENTS_FILTER_FROM_HOURS),
                                                    "to": ts + (60 * 60 * EVENTS_FILTER_TO_HOURS),
                                                    "serviceId": 154,  # "DC outage"
                                                    "environmentId": 204,  # "drills"
                                                    "type": "maintenance",
                                                    "severity": "major",
                                                },
                                                timeout=15)

        try:
            response.raise_for_status()
        except requests.exceptions.HTTPError:
            raise InfraEventsCacherError("Can't fetch events from infra.yandex-team.ru")
        try:
            raw_events = response.json()
        except ValueError:
            raise InfraEventsCacherError("Can't parse response from infra.yandex-team.ru as JSON")

        infra_events = []
        for event in raw_events:
            if event.get("is_deleted"):
                continue
            dcs = [
                dc
                for dc in ["sas", "man", "vla", "iva", "myt"]
                if event.get(dc)
            ]
            infra_event = InfraEvent(event["id"], dcs, event["start_time"], event["finish_time"])
            infra_events.append(dataclasses.asdict(infra_event))
        return infra_events
