import requests
import typing

from enum import IntEnum

from infra.rtc_sla_tentacles.backend.lib.harvesters.base import Harvester
from infra.rtc_sla_tentacles.backend.lib.util import requests_retry_session
from infra.rtc_sla_tentacles.backend.lib.funccall_stats_server import server as stat_server


class StateKind(IntEnum):
    ACTUAL = 0
    DOWNTIME_FORCE_OK = 1
    FLAPPING = 2
    NO_DATA_FORCE_CRIT = 3
    OTHER = 99


class Status(IntEnum):
    OK = 0
    WARN = 1
    CRIT = 2
    OTHER = 99


class Description(IntEnum):
    OK = 0
    TIMESTAMP_TOO_OLD = 10
    RESOURCE_ERROR = 20
    ICMPPING_DOWN = 30
    INVALID_HOSTNAME = 40
    HTTP_CONNECT_ERROR = 50
    NO_DATA = 60
    OTHER = 99


SERVICE_NAME = "timestamp-age"


class JugglerHarvester(Harvester):
    harvester_type = "juggler"
    _type_config_schema = {
        "type": "object",
        "properties": {
            "common_parameters": {
                "type": "object",
                "properties": {
                    "complete_events_url_essential_params": {
                        "type": "object",
                        "title": "Juggler additional URL params",
                    },
                    "complete_events_url": {
                        "type": "string",
                        "minLength": 1,
                        "title": "Juggler API URL",
                        "examples": ["http://juggler-api.search.yandex.net/api/events/complete_events"],
                    },
                    "timeout": {
                        "type": "integer",
                        "minimum": 1,
                        "title": "Request timeout",
                        "examples": [15],
                    }
                },
                "required": ["complete_events_url_essential_params", "complete_events_url", "timeout"],
            },
        },
        "required": ["common_parameters"],
    }

    def extract(self, ts: int) -> typing.Any:
        host_service_params = {
            "host_name": self.name,
            "service_name": SERVICE_NAME,
        }
        params = {**self.common_parameters["complete_events_url_essential_params"], **host_service_params}
        session = requests.session()
        session.headers["Content-Type"] = "application/json"
        session.headers["Accept"] = "application/json"
        session_with_retries = requests_retry_session(session=session)
        with stat_server.juggler_timing():
            response = session_with_retries.get(
                self.common_parameters["complete_events_url"], params=params, timeout=self.common_parameters["timeout"]
            )
        return response.json()

    def transform(self, ts: int, raw_data: typing.Any) -> typing.Tuple[typing.Any, typing.Any]:
        host_name = self.name
        service_name = SERVICE_NAME

        count_state_kind = {elem.name: 0 for elem in StateKind}
        count_status = {elem.name: 0 for elem in Status}
        count_description = {elem.name: 0 for elem in Description}

        values = []
        children = raw_data[host_name][service_name]["description"][0].get("children", {})
        for child in children.values():
            service_name_data = child[service_name]
            item = list(service_name_data.values())[0]
            item_compacted = self._compact_item(item)

            values.append(item_compacted)
            count_state_kind[item_compacted["state_kind"].name] += 1
            count_status[item_compacted["status"].name] += 1
            count_description[item_compacted["description"].name] += 1

        data = {"values": values}
        meta = {
            "count_values_types": {
                "state_kind": count_state_kind,
                "status": count_status,
                "description": count_description
            }
        }
        return meta, data

    @staticmethod
    def _compact_item(item):
        fqdn = item["host_name"]
        state_kind = item["state_kind"]
        status = item["status"]
        description = item["description"]

        item_compacted = {}
        item_compacted["fqdn"] = fqdn

        # Compact `state_kind`.
        item_compacted["state_kind"] = {
            'actual': StateKind.ACTUAL,
            'downtime_force_ok': StateKind.DOWNTIME_FORCE_OK,
            'flapping': StateKind.FLAPPING,
            'no_data_force_crit': StateKind.NO_DATA_FORCE_CRIT,
        }.get(state_kind, StateKind.OTHER)

        # Compact `status`.
        item_compacted["status"] = {
            'OK': Status.OK,
            'WARN': Status.WARN,
            'CRIT': Status.CRIT,
        }.get(status, Status.OTHER)

        # Compact `description`.
        description_compactification_map = {
            "http: body 'Ok'": Description.OK,
            "http: response: bad response code 418, body 'Timestamp too old'": Description.TIMESTAMP_TOO_OLD,
            "http: body 'ERROR: ": Description.RESOURCE_ERROR,
            "icmpping: down": Description.ICMPPING_DOWN,
            "Invalid hostname given": Description.INVALID_HOSTNAME,
            "http: connect:": Description.HTTP_CONNECT_ERROR,
            "NO DATA": Description.NO_DATA
        }
        item_compacted["description"] = Description.OTHER
        for description_value, compacted_value in description_compactification_map.items():
            if description_value in description:
                item_compacted["description"] = compacted_value
                break

        return item_compacted
