import typing

from infra.rtc_sla_tentacles.backend.lib.harvesters.base import Harvester
from infra.rtc_sla_tentacles.backend.lib.harvesters.dummy import DummyHarvester
from infra.rtc_sla_tentacles.backend.lib.harvesters.hq import HarvesterHQ
from infra.rtc_sla_tentacles.backend.lib.harvesters.infra_events_cacher import InfraEventsCacher
from infra.rtc_sla_tentacles.backend.lib.harvesters.juggler import JugglerHarvester
from infra.rtc_sla_tentacles.backend.lib.harvesters.nanny_state_dumper import NannyStateDumper
from infra.rtc_sla_tentacles.backend.lib.harvesters.resource_maker import ResourceMaker
from infra.rtc_sla_tentacles.backend.lib.harvesters.walle import WalleHarvester
from infra.rtc_sla_tentacles.backend.lib.harvesters.gencfg import GencfgHarvester
from infra.rtc_sla_tentacles.backend.lib.harvesters.yp_lite_allocation import YpLiteAllocationHarvester
from infra.rtc_sla_tentacles.backend.lib.harvesters.yp_lite_pod_count_tuner import YpLitePodCountTuner
from infra.rtc_sla_tentacles.backend.lib.harvesters.yp_lite_pods_tracker import YpLitePodsTracker
from infra.rtc_sla_tentacles.backend.lib.harvesters.yp_lite_reallocator import YpLiteReallocator
from infra.rtc_sla_tentacles.backend.lib.harvesters.yp_lite_switcher import YpLiteSwitcher
from infra.rtc_sla_tentacles.backend.lib.harvesters.clickhouse_dumper import ClickhouseDumper
from infra.rtc_sla_tentacles.backend.lib.harvesters.clickhouse_dropper import ClickhouseDropper
from infra.rtc_sla_tentacles.backend.lib.harvesters.clickhouse_optimizer import ClickhouseOptimizer
from infra.rtc_sla_tentacles.backend.lib.harvesters.monitoring.metrics_calculation import MetricsCalculation
from infra.rtc_sla_tentacles.backend.lib.harvesters.monitoring.missing_nodes import MissingNodesMonitoring
from infra.rtc_sla_tentacles.backend.lib.harvesters.monitoring.yp_lite_pods import YpLitePodsMonitoring
from infra.rtc_sla_tentacles.backend.lib.harvesters.monitoring.yp_unused_nodes import YpUnusedNodesMonitoring

from infra.rtc_sla_tentacles.backend.lib.harvesters_snapshots.manager import HarvesterSnapshotManager

from infra.rtc_sla_tentacles.backend.lib.config.interface import ConfigInterface


class HarvesterGroup:
    """Group of the harvesters of the same type"""

    def __init__(self, harvester_type: str, snapshot_manager: HarvesterSnapshotManager,
                 config_interface: ConfigInterface):
        self.harvester_type = harvester_type
        self._config_interface = config_interface
        self._snapshot_manager = snapshot_manager
        self.harvesters: typing.Dict[str, Harvester] = {}

        klass = self.get_harvester_type()
        if not klass:
            raise TypeError(f"Unknown harvester type: {self.harvester_type!r}")

        type_config = self._config_interface.get_harvester_config(self.harvester_type)

        for name in klass.secrets_map.keys():
            klass.secrets_map[name] = self._config_interface.secrets_config.get_secret(name)

        for instance in klass.build_instances(self._config_interface, self._snapshot_manager, type_config):
            self.harvesters[instance.name] = instance

    def get_harvester_type(self):
        return HARVESTERS.get(self.harvester_type)

    def get_harvesters(self) -> typing.Iterable[Harvester]:
        yield from self.harvesters.values()


class HarvestersManager:
    def __init__(self, config_interface: ConfigInterface, snapshot_manager: HarvesterSnapshotManager):
        self._config_interface = config_interface
        self._snapshot_manager = snapshot_manager
        self._harvester_groups: typing.Dict[str, HarvesterGroup] = {}

        harvester_groups = self._config_interface.get_configured_harvesters()

        for group in harvester_groups:
            self._harvester_groups[group] = HarvesterGroup(
                group,
                snapshot_manager=self._snapshot_manager,
                config_interface=self._config_interface,
            )

    def get_harvester_groups(self) -> typing.Iterable[HarvesterGroup]:
        return list(self._harvester_groups.values())

    def get_harvesters(self) -> typing.Iterable[Harvester]:
        for group in self._harvester_groups.values():
            yield from group.get_harvesters()


HARVESTERS = {
    harvester.harvester_type: harvester
    for harvester in (
        ClickhouseDumper,
        ClickhouseDropper,
        ClickhouseOptimizer,
        DummyHarvester,
        HarvesterHQ,
        InfraEventsCacher,
        JugglerHarvester,
        NannyStateDumper,
        ResourceMaker,
        WalleHarvester,
        YpLitePodCountTuner,
        YpLitePodsTracker,
        YpLiteReallocator,
        YpLiteSwitcher,
        GencfgHarvester,
        YpLiteAllocationHarvester,
        MetricsCalculation,
        MissingNodesMonitoring,
        YpLitePodsMonitoring,
        YpUnusedNodesMonitoring,
    )
}
