import datetime
import time
import typing

import humanize

from infra.rtc_sla_tentacles.backend.lib.harvesters.base import Harvester
from infra.rtc_sla_tentacles.backend.lib.clickhouse import database
from infra.rtc_sla_tentacles.backend.lib.clickhouse.client import ClickhouseClient
from infra.rtc_sla_tentacles.backend.lib.harvesters.infra_events_cacher import InfraEvent
from infra.rtc_sla_tentacles.backend.lib.metrics import metrics_provider, types as metrics_types, checks as slo_checks
from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager import util as checks_util


def datetime_repr(ts):
    now = int(time.time())
    dt = datetime.datetime.fromtimestamp(ts)
    return f"{(dt.strftime('%Y-%m-%d %H:%M'))} ({humanize.naturaldelta(datetime.timedelta(seconds=now - ts))} ago)"


class MetricsCalculation(Harvester):
    harvester_type = "metrics_calculation"

    def _get_dc_maintenance_events_in_progress(self, configured_check: slo_checks.ConfiguredCheck, ts: int,
                                               allocation_zone_id: str,
                                               infra_events: typing.List[InfraEvent]) -> typing.List[InfraEvent]:
        location = self.config_interface.tentacles_groups_config.get_option(allocation_zone_id, "location")
        if location is None:
            return []
        dcs = ["iva", "myt"] if location == "msk" else [location]
        return [
            event
            for dc in dcs
            for event in infra_events
            if dc in event.dcs
               and event.start_time < ts < event.finish_time + configured_check.get_force_ok_after_infra_event_interval()
        ]

    def _load_infra_events(self) -> typing.List[InfraEvent]:
        try:
            events = self._snapshot_manager.read_last_snapshot_data("infra_events_cacher", "infra_events_cacher")
        except Exception as _exc:
            self.logger.error("Can not load cached events from infra.yandex-team.ru: %s" % _exc)
            return []

        result = []
        for event_dict in events:
            result.append(InfraEvent(**event_dict))

        self.logger.debug("Loaded %d cached events in infra.yandex-team.ru" % len(result))
        return result

    def _report(self, ok, configured_check, description):
        if ok:
            report_fn = self.juggler_sender.ok
        else:
            report_fn = self.juggler_sender.crit

        report_fn(
            host=configured_check.host,
            service=configured_check.service,
            description=description,
        )

    def extract(self, ts: int):
        clickhouse_client = ClickhouseClient(self.config_interface)

        last_iteration_ts = int(clickhouse_client.raw(database.get_last_timestamp(return_timestamp=True)))
        outdated_metrics = last_iteration_ts + metrics_provider.MONITORING_DATA_TTL < int(time.time())

        infra_events = self._load_infra_events()

        with self.time_it("clickhouse fetching"):
            all_zone_metrics = metrics_provider.get_all_allocation_zones_metrics(
                self.config_interface, clickhouse_client,
            )
        for allocation_zone_id in self.config_interface.get_allocation_zones():
            zone_metrics = all_zone_metrics[allocation_zone_id]
            links_msg = checks_util.get_allocation_zone_description_part(self.config_interface, allocation_zone_id)
            for configured_check in slo_checks.get_fast_configured_checks(self.config_interface, allocation_zone_id):
                events = self._get_dc_maintenance_events_in_progress(
                    configured_check, ts, allocation_zone_id, infra_events
                )
                if events:
                    self._report(
                        ok=True,
                        configured_check=configured_check,
                        description=("DC maintenance in progress or recently finished, check forced to OK.\n"
                                     f"Maintenance events:\n"
                                     f"{' '.join([event.to_link() for event in events])}"),
                    )
                    continue
                if outdated_metrics:
                    self._report(
                        ok=False,
                        configured_check=configured_check,
                        description=f"SLO harvesting stalled, last update was {datetime_repr(last_iteration_ts)}",
                    )
                    continue
                if configured_check.slo_type == metrics_types.SloType.availability:
                    availability_metrics = zone_metrics.availability
                    check_ok = availability_metrics.is_slo_ok(configured_check.limit)
                    description = (
                        f"Availability SLA "
                        f"{availability_metrics.number_of_available_tentacles}/"
                        f"{availability_metrics.number_of_not_excluded_tentacles}"
                        f"({availability_metrics.percent_of_available_tentacles:.2f}%). "
                        f"{configured_check.limit}% is border. "
                        f"{availability_metrics.number_of_excluded_tentacles} excluded.\n"
                    ) + links_msg
                elif configured_check.slo_type == metrics_types.SloType.redeployed_on_time:
                    redeployment_metrics = zone_metrics.redeployed_on_time
                    check_ok = redeployment_metrics.is_slo_ok(configured_check.limit)

                    session_str = ""
                    if last_success_redeployment_ts := redeployment_metrics.last_success_redeployment_end_ts:
                        session_str += (
                            f"Last successful redeploy was {datetime_repr(last_success_redeployment_ts)}.\n"
                        )
                    if current_session_ts := redeployment_metrics.current_redeployment_start_ts:
                        session_str += f"Current redeploy was started at {datetime_repr(current_session_ts)}.\n"

                    description = (
                        f"Redeployment "
                        f"{redeployment_metrics.number_of_tentacles_with_fresh_ts_resource}/"
                        f"{redeployment_metrics.number_of_not_excluded_tentacles}"
                        f"({redeployment_metrics.percent_of_tentacles_with_fresh_ts_resource:.2f}%). "
                        f"{configured_check.limit}% is border. "
                        f"{redeployment_metrics.number_of_excluded_tentacles} excluded.\n"
                        f"{session_str}"
                    ) + links_msg
                elif configured_check.slo_type == metrics_types.SloType.reallocation:
                    reallocation_metrics = zone_metrics.reallocation
                    check_ok = reallocation_metrics.is_slo_ok()

                    session_str = ""
                    if last_success_reallocation_ts := reallocation_metrics.last_success_reallocation_end_ts:
                        session_str += (
                            f"Last successful reallocation was {datetime_repr(last_success_reallocation_ts)}.\n"
                        )
                    if current_session_ts := reallocation_metrics.current_reallocation_start_ts:
                        session_str += f"Current reallocation was started at {datetime_repr(current_session_ts)}.\n"

                    description = session_str + links_msg
                else:
                    raise RuntimeError(f"unknown slo type {configured_check.slo_type}")
                self._report(check_ok, configured_check, description)
        return {
            allocation_zone_id: allocation_zone_metrics.to_dict()
            for allocation_zone_id, allocation_zone_metrics
            in all_zone_metrics.items()
        }
