import logging
import time
import typing

from infra.rtc_sla_tentacles.backend.lib.harvesters.base import Harvester
from infra.rtc_sla_tentacles.backend.lib.yp_lite.pods_manager import YpLitePodsManager


def _convert_yp_lite_pods_manager_monitoring_metrics_to_signals(monitoring_data: dict) -> typing.List:
    signals = []
    if not monitoring_data:
        return signals
    prefix = "podsmanager"
    postfix = "axxv"

    def _yield_stats_as_yasm_signals(_nanny_service_name, _stats):
        data_age_sec = int(time.time()) - _stats["data_freshness"]
        yield [
            f"{prefix}__{_nanny_service_name}__data_age_sec__{postfix}",
            data_age_sec
        ]
        for stats_group in ["pods", "nodes", "nanny_instances"]:
            for metric_name, metric_value in _stats[stats_group].items():
                yield [
                    f"{prefix}__{_nanny_service_name}__{metric_name}__{postfix}",
                    metric_value
                ]

    for nanny_service_name, nanny_service_data in monitoring_data.items():
        for signal in _yield_stats_as_yasm_signals(nanny_service_name, nanny_service_data["stats"]):
            signals.append(signal)
    return signals


class YpLitePodsMonitoring(Harvester):
    harvester_type = "yp_lite_pods_monitoring"

    def extract(self, ts: int):
        monitoring_data = {}
        harvester_config = self.config_interface.get_harvester_config("yp_lite_pods_tracker")
        for nanny_service_name in harvester_config["arguments"]:
            yp_cluster = self.config_interface.tentacles_groups_config.get_option(nanny_service_name, "yp_cluster")
            restrict_nodes_to_specified_in_podset = self.config_interface.tentacles_groups_config.get_option(
                nanny_service_name, "restrict_nodes_to_specified_in_podset")
            monitoring_data[nanny_service_name] = {}
            monitoring_data[nanny_service_name]["config"] = {
                "nanny_service_name": nanny_service_name,
                "yp_cluster": yp_cluster,
                "restrict_nodes_to_specified_in_podset": restrict_nodes_to_specified_in_podset,
            }
            with YpLitePodsManager(
                nanny_service_name=nanny_service_name,
                yp_cluster=yp_cluster,
                logger=logging.getLogger(),
                restrict_nodes_to_specified_in_podset=restrict_nodes_to_specified_in_podset,
                config_interface=self.config_interface,
                init_clickhouse_client=True,
            ) as pods_manager:
                pods_manager.load_pods_from_clickhouse()
                monitoring_data[nanny_service_name]["stats"] = pods_manager.get_stats()

            # TODO(rocco66): TENTACLES-327
            service_metrics_from_harvester = self._snapshot_manager.read_last_snapshot_data(
                "yp_lite_allocation", nanny_service_name,
                projection={
                    "data.pods_scheduling_errors": True,
                    "data.nodes_up_scheduling_errors": True,
                    "_id": False,
                },
            )
            if service_metrics_from_harvester:
                monitoring_data[nanny_service_name]["stats"]["pods"].update(service_metrics_from_harvester)
        return {
            "yasm_pods_manager_metrics": _convert_yp_lite_pods_manager_monitoring_metrics_to_signals(monitoring_data),
        }
