import dataclasses
import datetime
import humanize

from infra.rtc_sla_tentacles.backend.lib.clickhouse import database
from infra.rtc_sla_tentacles.backend.lib.clickhouse.client import ClickhouseClient
from infra.rtc_sla_tentacles.backend.lib.harvesters.base import Harvester
from infra.rtc_sla_tentacles.backend.lib.metrics import metrics_provider, checks as slo_checks
from infra.rtc_sla_tentacles.backend.lib.yp_lite.pods_manager import NodeInfo
from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager import util as checks_util

MONITORING_BORDER = 24 * 60 * 60


# TODO(rocco66): remove it https://st.yandex-team.ru/TENTACLES-389
class YpUnusedNodesMonitoring(Harvester):
    harvester_type = "yp_unused_nodes_monitoring"

    def extract(self, ts: int):
        allocation_zone_id = self.name
        query = f"""
        SELECT DISTINCT fqdn
        FROM {database.Tentacle.table_name()}
        WHERE
          ts > '{database.to_clickhouse_datetime(ts - MONITORING_BORDER)}'
          AND nanny_service_name = '{allocation_zone_id}'
          AND slo_redeployment_fresh_ts_resource == 1
        """
        client = ClickhouseClient(self.config_interface)
        live_nodes = set(r.fqdn for r in client.select(query))

        # TODO(rocco66): TENTACLES-327
        service_metrics_from_harvester = self._snapshot_manager.read_last_snapshot_data(
            "yp_lite_allocation", allocation_zone_id,
            projection={
                "data.nodes_info": True,
                "_id": False,
            },
        )
        if not service_metrics_from_harvester:
            raise RuntimeError("There is no information about allocation zone")
        unused_nodes = set()
        nodes_info_dump = service_metrics_from_harvester["nodes_info"]
        if not nodes_info_dump:
            raise RuntimeError(f"Allocation zone without nodes {allocation_zone_id}")
        for node_info in (NodeInfo(*n) for n in nodes_info_dump):
            if node_info.node_id not in live_nodes:
                unused_nodes.add(node_info.node_id)

        configured_check = slo_checks.get_yp_unused_nodes_slow_configured_check(
            self.config_interface, allocation_zone_id
        )
        percent_limit = self.config_interface.tentacles_groups_config.get_unused_nodes_limit(allocation_zone_id)
        sorted_unused_nodes = sorted(unused_nodes)
        metric = metrics_provider.MetricsTentaclesYpUnusedNodes(
            len(unused_nodes), len(nodes_info_dump), percent_limit, sorted_unused_nodes,
        )
        description = (
            f"YP unused nodes {len(unused_nodes)}/{len(nodes_info_dump)} ({metric.get_percent_of_unused_nodes():.2f}%). "
            f"Observable interval is {humanize.naturaldelta(datetime.timedelta(seconds=MONITORING_BORDER))}.\n"
        )
        if sorted_unused_nodes:
            description += f"Some nodes: {', '.join(sorted_unused_nodes[:5])}\n"
        description += checks_util.get_allocation_zone_description_part(
            self.config_interface, allocation_zone_id
        )
        if metric.is_slo_ok():
            report_fn = self.juggler_sender.ok
        else:
            report_fn = self.juggler_sender.crit
        report_fn(
            host=configured_check.host,
            service=configured_check.service,
            description=description,
        )
        return {
            "metric": dataclasses.asdict(metric),
        }
