import typing
import requests
from urllib.parse import urljoin

from infra.rtc_sla_tentacles.backend.lib.clickhouse.client import ClickhouseClient
from infra.rtc_sla_tentacles.backend.lib.clickhouse.database import NannyState
from infra.rtc_sla_tentacles.backend.lib.harvesters.base import Harvester
from infra.rtc_sla_tentacles.backend.lib.util import requests_retry_session
from infra.rtc_sla_tentacles.backend.lib.tentacle_agent import const as agent_const
from infra.rtc_sla_tentacles.backend.lib.funccall_stats_server import server as stat_server


class NannyStateDumperException(Exception):
    pass


class NannyStateDumper(Harvester):
    secrets_map = {
        "NANNY_OAUTH_TOKEN": None
    }
    harvester_type = "nanny_state_dumper"

    def extract(self, ts: int) -> typing.Any:
        session = requests.session()
        session.headers["Content-Type"] = "application/json"
        session.headers["Accept"] = "application/json"
        session.headers["Authorization"] = "OAuth {}".format(self.secrets_map["NANNY_OAUTH_TOKEN"])
        session_with_retries = requests_retry_session(session=session)
        current_state = self._get_current_state(session=session_with_retries)
        active_revision_id = self._get_active_revision_id(session_with_retries)
        current_active_snapshot = self._get_active_snapshot_data(session_with_retries,
                                                                 current_state,
                                                                 active_revision_id)

        mongo_result = {
            "current_active": current_active_snapshot,
        }

        latest_snapshot_data = self._get_latest_snapshot_data(current_state["content"]["active_snapshots"])
        if latest_snapshot_data is None:
            raise NannyStateDumperException("Can not parse latest snapshot data")

        clickhouse_result = {
            "ts": ts,
            "nanny_service_name": self.name,
            "deploy_gencfg": 1 if not self.name.startswith("yp_") else 0,
            "deploy_yp_lite": 1 if self.name.startswith("yp_") else 0,
            "current_state": current_state["content"]["summary"]["value"],
            "reallocation_id": current_state["reallocation"]["id"],
            "reallocation_state_status": self._parse_reallocation_status((current_state["reallocation"]
                                                                                       ["state"]
                                                                                       ["status"])),
            "reallocation_taskgroup_id": current_state["reallocation"]["taskgroup_id"],
            "latest_snapshot_id": latest_snapshot_data.get("snapshot_id"),
            "latest_snapshot_taskgroup_id": latest_snapshot_data.get("taskgroup_id")
        }

        return {
            "mongodb_result": mongo_result,
            "clickhouse_result": clickhouse_result
        }

    def transform(self, ts: int, data) -> typing.Tuple[typing.Any, typing.Any]:
        self._dump_to_clickhouse(data["clickhouse_result"])
        meta: dict = {}
        return meta, data["mongodb_result"]

    def _dump_to_clickhouse(self, _data: dict):
        output = [NannyState(**_data)]
        client = ClickhouseClient(self.config_interface)
        client.insert(output)

    def _get_current_state(self, session):
        url = urljoin(self.common_parameters["nanny_api"],
                      self.common_parameters["nanny_current_state_url_template"].format(self.name))
        with stat_server.nanny_timing():
            return session.get(url, timeout=self.common_parameters["timeout"]).json()

    def _get_active_revision_id(self, session):
        url = urljoin(self.common_parameters["nanny_api"],
                      self.common_parameters["nanny_active_revision_id_url_template"].format(self.name))
        with stat_server.nanny_timing():
            return session.get(url, timeout=self.common_parameters["timeout"]).json()["active_revision_id"]

    def _get_active_snapshot_data(self, session, current_state, revision_id):
        active_snapshot_id = None
        for snapshot in current_state["content"]["active_snapshots"]:
            if snapshot["conf_id"] == revision_id:
                active_snapshot_id = snapshot["snapshot_id"]

        url = urljoin(self.common_parameters["nanny_api"],
                      self.common_parameters["nanny_snapshot_resources_url_template"].format(self.name))
        with stat_server.nanny_timing():
            snapshot = session.get(url, timeout=self.common_parameters["timeout"]).json()

        if snapshot.get("error"):
            raise RuntimeError(f"Nanny snapshot fetch error: {snapshot['error']}/{snapshot['msg']}")
        for resource in snapshot["content"]["resources"]["url_files"]:
            if resource["local_path"] == agent_const.AGENT_RESOURCE_NAME:
                return {
                    "snapshot_id": active_snapshot_id,
                    "timestamp_rbtorrent_id": resource["url"],
                }

    @staticmethod
    def _get_latest_snapshot_data(snapshots):
        """
            Choose snapshot with latest timestamp in "conf_id".
            Example of "conf_id": "rtc_sla_tentacles_testing_sas-1566460064577".
        """
        result = None
        result_conf_ts = 0
        for snapshot in snapshots:
            snapshot_conf_ts = int(snapshot["conf_id"].split("-")[1])
            if snapshot_conf_ts > result_conf_ts:
                result = snapshot
                result_conf_ts = snapshot_conf_ts
        return result

    @staticmethod
    def _parse_reallocation_status(status):
        return {
            "In progress": "IN_PROGRESS",
            "Idle": "IDLE",
            "Error": "ERROR"
        }.get(status, "OTHER")
