import glob
import os
import shutil
import subprocess
import time
import typing
import json
import pathlib

from infra.rtc_sla_tentacles.backend.lib.harvesters.base import Harvester
from infra.rtc_sla_tentacles.backend.lib.harvesters import nanny_state_dumper
from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager import harvesters_snapshots_freshness_checks
from infra.rtc_sla_tentacles.backend.lib.tentacle_agent import const as agent_const


class ResourceMaker(Harvester):
    harvester_type = 'resource_maker'
    run_on_all_workers = True

    _RESOURCE_META_FILE = "meta.json"

    _instance_config_schema = {
        "type": "object",
        "properties": {
            "storage_dir": {
                "type": "string",
                "title": "Place to store archives",
                "examples": ["/tmp/"],
            },
            "keep_copies": {
                "type": "integer",
                "title": "Amount of old copies to keep",
                "minimum": 0,
            },
        },
        "required": ["storage_dir", "keep_copies"],
    }

    def _get_folders_for_cleanup(self, storage_dir, keep):
        candidates = glob.glob(f'{storage_dir}/*')
        candidates.sort(key=storage_sorting_key)

        labels = self._snapshot_manager.get_last_snapshot_labels_from_harvester(
            nanny_state_dumper.NannyStateDumper.harvester_type
        )
        nanny_data = []
        for harvester_labels in labels:
            nanny_data.append(self._snapshot_manager.read_snapshot(label=harvester_labels))
        actual_resources = {
            harvester_snapshot.data["current_active"]["timestamp_rbtorrent_id"]
            for harvester_snapshot in nanny_data
        }

        keep_counter = 0
        for candidate_index, candidate in enumerate(candidates):
            meta_file_path = os.path.join(candidate, self._RESOURCE_META_FILE)
            if not pathlib.Path(meta_file_path).exists():
                yield candidate
                continue
            if os.path.getmtime(meta_file_path) > time.time() - 2 * 60 * 60:
                continue
            with open(meta_file_path) as meta_file:
                resource_meta = json.load(meta_file)
            if resource_meta["resource_id"] in actual_resources:
                keep_counter += 1
                continue
            if (len(candidates) - candidate_index) > (keep - keep_counter):
                yield candidate
            else:
                return

    def cleanup_old_copies(self, storage_dir: str, keep: int) -> None:
        for candidate in self._get_folders_for_cleanup(storage_dir, keep):
            self.logger.info("will remove old resource %s", candidate)
            if os.path.exists(candidate):
                try:
                    shutil.rmtree(candidate)
                except Exception:
                    self.logger.exception("failed to remove %s", candidate)

    def _get_resource_period(self):
        return int(self.get_interval() * 1.2)

    def _get_align_ts(self):
        # NOTE(rocco66): use different alignment for assured resource generation every minute
        now = int(time.time())
        return now - now % self._get_resource_period()

    def extract(self, ts: int) -> typing.Tuple[str, int]:
        storage_dir = self.arguments['storage_dir']
        if not os.path.exists(storage_dir):
            os.makedirs(storage_dir)

        try:
            self.cleanup_old_copies(storage_dir, self.arguments['keep_copies'])
        except Exception:
            self.logger.exception("Cleanup error")

        align_ts = self._get_align_ts()
        target_dir = os.path.join(storage_dir, str(align_ts))
        if not os.path.exists(target_dir):
            os.makedirs(target_dir)

        resource_name = agent_const.AGENT_RESOURCE_NAME
        with open(os.path.join(target_dir, resource_name), 'w') as resource_file:
            resource_file.write(str(align_ts))

        # Run sky share
        process = subprocess.Popen(
            ["sky", "share", '-d', target_dir, resource_name],
            shell=False,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
        )
        try:
            stdout, stderr = process.communicate(timeout=15)
        except subprocess.TimeoutExpired:
            process.kill()
            process.wait(timeout=15)
            raise

        self.logger.debug("sky share stdout: %r", stdout)

        if stdout.startswith(b"rbtorrent:"):
            resource_id = stdout.decode('utf-8').rstrip()
            with open(os.path.join(target_dir, self._RESOURCE_META_FILE), "w") as meta_file:
                json.dump({"resource_id": resource_id}, meta_file)
            return resource_id, align_ts
        else:
            raise ValueError(f"Failed to share: {stderr.strip()}")

    def transform(self, ts: int, raw_data: typing.Tuple[str, int]) -> typing.Tuple[dict, dict]:
        data = {
            'resource_id': raw_data[0],
            'resource_timestamp': raw_data[1],
            'tar': False,
        }
        meta = {'success': True}
        self.juggler_sender.ok(service=harvesters_snapshots_freshness_checks.RESOUCE_MAKING_PROCESS_SERVICE)
        return meta, data


def storage_sorting_key(path: str) -> int:
    try:
        return int(os.path.basename(path))
    except Exception:
        return 0
