# encoding: utf-8
import logging
import time
import concurrent.futures

import pymongo

from infra.rtc_sla_tentacles.backend.lib.config.interface import ConfigInterface
from infra.rtc_sla_tentacles.backend.lib.harvesters.manager import HarvestersManager
from infra.rtc_sla_tentacles.backend.lib.funccall_stats_server import server as stat_server
from infra.rtc_sla_tentacles.backend.lib.mongo.client import MongoClient


logger = logging.getLogger("harvesters.ticker")


class Ticker(object):

    DEFAULT_WORKER_COUNT = 10

    def __init__(self, harvesters_manager: HarvestersManager, config_interface: ConfigInterface,
                 mongo_client: MongoClient, stats_server: stat_server.FunccallStatsServer,
                 worker_count=DEFAULT_WORKER_COUNT):
        self.harvesters_manager = harvesters_manager
        self._stats_server = stats_server
        harvesters_config = config_interface.get_harvesters_results_storage_config()
        mongo_database_handler = mongo_client.get_database(None)
        self.mongo_collection = mongo_database_handler[harvesters_config["locks_collection_name"]]
        self._local_start_times = {}
        self._init_locks()
        self._workers_pool = concurrent.futures.ThreadPoolExecutor(max_workers=worker_count)

    def _init_locks(self):
        for harvester in self.harvesters_manager.get_harvesters():
            if not harvester.run_on_all_workers:
                try:
                    self.mongo_collection.insert_one(
                        {
                            "_id": self._get_harvester_key(harvester),
                            "next_ts": 0,
                        }
                    )
                except pymongo.errors.DuplicateKeyError:
                    pass  # NOTE(rocco66): this is Ok

    @staticmethod
    def _get_harvester_key(harvester):
        return f"{harvester.harvester_type}:{harvester.name}"

    def _try_to_lock(self, harvester, iteration_ts) -> bool:
        key = self._get_harvester_key(harvester)
        response = self.mongo_collection.update_one(
            {
                "_id": key,
                "next_ts": {
                    "$lte": iteration_ts,
                }
            },
            {
                "$set": {
                    "next_ts": iteration_ts + harvester.get_interval()
                }
            },
        )
        result = bool(response.modified_count)
        logger.info(f"Try to lock {key} {result}")
        return result

    def _run_with_stats(self, harvester, started_time):
        try:
            harvester.run(started_time)
            stat_server.add_ok(harvester.harvester_type, time.time() - started_time)
        except Exception as exc:
            stat_server.add_error(harvester.harvester_type, time.time() - started_time)
            logger.exception(f"Uncaught error for {self._get_harvester_key(harvester)}")
            raise exc
        finally:
            self._stats_server.harvester_free()

    def _run(self, harvester, started_time):
        self._stats_server.harvester_acquire()
        self._workers_pool.submit(self._run_with_stats, harvester, started_time)

    @staticmethod
    def _get_iteration_time(harvester) -> int:
        now = int(time.time())
        return now - (now % harvester.get_interval())  # NOTE(rocco66): old ticker behaviour

    def _tick(self) -> int:
        harvesters = self._get_actual_harvesters()
        logger.info(f"{len(harvesters)} harvesters are actual")
        started_harvester_counter = 0
        for harvester_index, harvester in enumerate(harvesters):
            iteration_ts = self._get_iteration_time(harvester)
            if harvester.run_on_all_workers:
                # NOTE(rocco66): do not round time here or several harvester starts will be
                self._local_start_times[self._get_harvester_key(harvester)] = time.time()

                self._run(harvester, iteration_ts)
                started_harvester_counter += 1
            else:
                if self._stats_server.get_free_harvesters() > 0:
                    if self._try_to_lock(harvester, iteration_ts):
                        self._run(harvester, iteration_ts)
                        started_harvester_counter += 1
        return started_harvester_counter

    def loop(self):
        while True:
            try:
                started_harvester = self._tick()
                if not started_harvester:
                    logger.info("No more actual harvesters, sleep")
                    time.sleep(5)
            except Exception:
                logger.exception("Uncaught error in main worker loop")
                time.sleep(1)

    def _get_actual_harvesters(self):
        result = []
        all_locks = {lock["_id"]: lock["next_ts"] for lock in self.mongo_collection.find({})}
        for harvester in self.harvesters_manager.get_harvesters():
            if harvester.run_on_all_workers:
                prev_start = self._local_start_times.get(self._get_harvester_key(harvester))
                now = time.time()
                if not prev_start:
                    result.append(harvester)
                elif prev_start + harvester.get_interval() <= now + 1:
                    result.append(harvester)
            else:
                harvester_lock_key = self._get_harvester_key(harvester)
                next_ts = all_locks.get(harvester_lock_key)
                if not next_ts or next_ts < time.time() + 1:
                    result.append(harvester)
        return result
