import dataclasses
import typing

import walle_api

from infra.rtc_sla_tentacles.backend.lib.harvesters.base import Harvester
from infra.rtc_sla_tentacles.backend.lib import util
from infra.rtc_sla_tentacles.backend.lib.funccall_stats_server import server as stat_server


TARGET_TAG = "tentacles_monitored"


def get_needed_tags(tags):
    return [tag for tag in tags
            if tag.startswith("rtc.stage-") or tag.startswith("rtc.prj-") or tag.startswith("rtc.scheduler-")]


@dataclasses.dataclass
class WalleHost:
    fqdn: str
    walle_project: str
    walle_location_short_datacenter_name: str
    walle_operation_state: str
    walle_state: str
    walle_status: str
    rtc_stage: typing.List[str]
    deploy_engines: typing.List[str]
    walle_health_status: typing.Union[str, None]
    walle_health_fail_reasons: typing.List
    scenario_id: int
    ticket: str


class WalleHarvester(Harvester):
    secrets_map = {
        "WALLE_OAUTH_TOKEN": None
    }
    harvester_type = "walle"

    _FETCHED_FILEDS = [
        "location.short_datacenter_name",
        "operation_state",
        "state",
        "status",
        "health.status",
        "health.reasons",
        "name",
        "project",
        "tags",
        "inv",  # NOTE(rocco66): for cursor (see walle SDK)
        "scenario_id",
        "ticket",
    ]

    @staticmethod
    def _parse_raw_host(raw_host):
        deploy_engines = []
        rtc_stage = None

        for tag in get_needed_tags(raw_host["tags"]):
            key, value = tag.split("-", 1)
            if key == "rtc.stage":
                rtc_stage = value
            elif key == "rtc.scheduler":
                deploy_engines.append(value)
        if not deploy_engines:
            deploy_engines = ["gencfg"]
        walle_host = WalleHost(
            fqdn=raw_host["name"],
            walle_project=raw_host["project"],
            walle_location_short_datacenter_name=raw_host.get("location", {}).get("short_datacenter_name") or "",
            walle_operation_state=raw_host.get("operation_state"),
            walle_state=raw_host.get("state"),
            walle_status=raw_host.get("status"),
            rtc_stage=rtc_stage,
            deploy_engines=deploy_engines,
            walle_health_status=None,
            walle_health_fail_reasons=[],
            scenario_id=raw_host.get("scenario_id", 0),
            ticket=raw_host.get("ticket", ""),
        )
        health = raw_host.get("health", {})
        if health:
            walle_host.walle_health_status = health.get("status")
            if "reasons" in health:
                walle_host.walle_health_fail_reasons = health.get("reasons")
        return walle_host

    def retry_iter_hosts(self, client, **kwargs):
        cursor = 0
        retry_get_hosts = util.retry()(client.get_hosts)
        while True:
            result = retry_get_hosts(
                fields=self._FETCHED_FILEDS,
                limit=10000,
                cursor=cursor,
                **kwargs
            )
            hosts = result["result"]
            if not hosts:
                break

            for host in hosts:
                del host["inv"]
                yield host

            if "next_cursor" in result:
                cursor = result["next_cursor"]
            else:
                break

    def extract(self, ts):
        client = walle_api.WalleClient(access_token=self.secrets_map["WALLE_OAUTH_TOKEN"])
        with stat_server.walle_timing():
            hosts = list(self.retry_iter_hosts(client, tags=TARGET_TAG))
        # NOTE(rocco66): filter nameless hosts https://st.yandex-team.ru/TENTACLES-388
        return [self._parse_raw_host(h) for h in hosts if h.get("name")]

    def transform(self, ts, data):
        meta = {}
        if data:
            return meta, {"values": [dataclasses.astuple(h) for h in data]}
        else:
            return meta, {"values": None}
