import dataclasses

from infra.rtc_sla_tentacles.backend.lib.config.exceptions import ConfigGetOptionError
from infra.rtc_sla_tentacles.backend.lib.harvesters.base import Harvester
from infra.rtc_sla_tentacles.backend.lib.yp_lite.pods_manager import YpLitePodsManager


@dataclasses.dataclass
class PodInfo:
    pod_id: str
    node_id: str
    node_hfsm_state: str
    scheduling_error: str


def _node_is_up(node_id, node_index):
    node_info = node_index.get(node_id)
    if node_info:
        return node_info.hfsm_state == "up"
    return False


class YpLiteAllocationHarvester(Harvester):
    harvester_type = "yp_lite_allocation"

    def extract(self, ts: int):
        allocation_zone_id = self.name
        yp_cluster = self.config_interface.tentacles_groups_config.get_option(allocation_zone_id, "yp_cluster")

        try:
            restrict_nodes_to_specified_in_podset = self.config_interface.tentacles_groups_config.get_option(
                allocation_zone_id, "restrict_nodes_to_specified_in_podset")
        except ConfigGetOptionError:
            restrict_nodes_to_specified_in_podset = False

        with YpLitePodsManager(nanny_service_name=allocation_zone_id,
                               yp_cluster=yp_cluster,
                               logger=self.logger,
                               restrict_nodes_to_specified_in_podset=restrict_nodes_to_specified_in_podset,
                               config_interface=self.config_interface,
                               init_yp_client=True) as pods_manager:
            nodes_index = pods_manager.fetch_yp_nodes_info()
            pods_info = []
            pods_scheduling_errors = 0
            nodes_up_scheduling_errors = 0
            for pod_id, error, hints, scheduling_node in pods_manager.select_objects(
                "pod",
                filter=f'[/meta/pod_set_id]="{pods_manager.get_podset_name()}"',
                selectors=[
                    "/meta/id",
                    "/status/scheduling/error",
                    "/spec/scheduling/hints",
                    "/status/scheduling/node_id",
                ],
            ):
                if hints:
                    node_id = hints[0]["node_id"]
                else:
                    node_id = scheduling_node or ""
                if node_info := nodes_index.get(node_id):
                    node_hfsm_state = node_info.hfsm_state
                else:
                    node_hfsm_state = ""
                if error:
                    pods_scheduling_errors += 1
                    if _node_is_up(node_id, nodes_index):
                        nodes_up_scheduling_errors += 1
                    try:
                        first_level = error["inner_errors"][0]
                        second_level = first_level.get("inner_errors")
                        if second_level:
                            error_str = second_level[0]["message"]
                        else:
                            error_str = first_level["message"]
                    except Exception:
                        self.logger.exception("pod hint error")
                        # NOTE(rocco66): too scary, fallback
                        error_str = error.get("message", "")
                else:
                    error_str = ""
                pods_info.append(dataclasses.astuple(
                    PodInfo(pod_id, node_id, node_hfsm_state, error_str))
                )

        scheduling_errors_limit = self.config_interface.tentacles_groups_config.get_scheduling_errors_limit(
            allocation_zone_id
        )
        # TODO(rocco66): calculate scheduling errors status here, instead of golovan signals
        # TODO(rocco66): join with monitoring/yp_lite_pods harvester
        return {
            "pods_info": pods_info,
            "nodes_total": len(nodes_index),
            "nodes_info": [dataclasses.astuple(node_info) for node_id, node_info in nodes_index.items()],
            "pods_scheduling_errors": pods_scheduling_errors,
            "nodes_up_scheduling_errors": nodes_up_scheduling_errors,
            "nodes_up_count": sum(node_info.hfsm_state == "up" for node_info in nodes_index.values()),
            "monitoring_min_percent_of_scheduling_errors": scheduling_errors_limit,
        }
