import typing

from infra.rtc_sla_tentacles.backend.lib.clickhouse.client import ClickhouseClient
from infra.rtc_sla_tentacles.backend.lib.config.interface import ConfigInterface
from infra.rtc_sla_tentacles.backend.lib.harvesters.base import Harvester
from infra.rtc_sla_tentacles.backend.lib.reroll_history.history import ReallocationHistory, RedeploymentHistory, \
    RerollHistoryException
from infra.rtc_sla_tentacles.backend.lib.yp_lite.allocation import AllocationRequestProvider

from nanny_rpc_client import RetryingRpcClient
from infra.nanny.yp_lite_api.proto import pod_reallocation_api_pb2
from infra.nanny.yp_lite_api.py_stubs.pod_sets_api_stub import YpLiteUIPodSetsServiceStub
from infra.nanny.yp_lite_api.py_stubs.pod_reallocation_api_stub import YpLiteReallocationServiceStub
from infra.rtc_sla_tentacles.backend.lib.nanny import nanny_client
from infra.rtc_sla_tentacles.backend.lib.funccall_stats_server import server as stat_server


# noinspection PyProtectedMember
class YpLiteReallocator(Harvester):
    harvester_type = 'yp_lite_reallocator'

    secrets_map = {
        'NANNY_OAUTH_TOKEN': None,
        'YP_TOKEN': None,
    }

    def extract(self, ts: int) -> typing.Any:
        if not self._allow_reallocation(ts, self.config_interface):
            return

        snapshot_id = self._get_nanny_snapshot_id()
        if not snapshot_id:
            self.logger.info("There is no snapshot ID in ClickHouse yet for %s", self.name)
            return

        self.arguments = self.config_interface.get_tentacles_group_reallocation_settings(self.name)

        nanny_pod_sets_rpc_client_config = self.config_interface.get_nanny_rpc_client_config(
            rpc_url_name="yp_lite_pod_sets_url")
        yp_lite_ui_podsets_service_stub = YpLiteUIPodSetsServiceStub(RetryingRpcClient(
            **nanny_pod_sets_rpc_client_config))

        # Get AllocationRequest() of a random pod.
        nanny_service_id = self.name
        yp_cluster = self.config_interface.tentacles_groups_config.get_option(nanny_service_id, "yp_cluster")
        allocation_request_provider = AllocationRequestProvider(
            nanny_service_id, yp_cluster, yp_lite_ui_podsets_service_stub
        )
        # To trigger reallocation we need a diff in old and new pod specs.
        allocation_request_provider.flip_root_fs_quota_megabytes()
        allocation_request = allocation_request_provider.get_allocation_request()

        # Create StartPodReallocationRequest().
        start_pod_reallocation_request = pod_reallocation_api_pb2.StartPodReallocationRequest()
        start_pod_reallocation_request.service_id = nanny_service_id
        start_pod_reallocation_request.snapshot_id = snapshot_id
        start_pod_reallocation_request.degrade_params.max_unavailable_pods = \
            self.arguments["degrade_params"]["max_unavailable_pods"]
        start_pod_reallocation_request.degrade_params.min_update_delay_seconds = \
            self.arguments["degrade_params"]["min_update_delay_seconds"]
        start_pod_reallocation_request.allocation_request.CopyFrom(allocation_request)

        # Get current reallocation ID.
        get_pod_reallocation_spec_request = pod_reallocation_api_pb2.GetPodReallocationSpecRequest()
        get_pod_reallocation_spec_request.service_id = nanny_service_id
        nanny_pod_reallocation_rpc_client_config = self.config_interface.get_nanny_rpc_client_config(
            rpc_url_name="yp_lite_pod_reallocation_url")
        yp_lite_reallocation_service_stub = YpLiteReallocationServiceStub(RetryingRpcClient(
            **nanny_pod_reallocation_rpc_client_config))
        with stat_server.yp_lite_timing():
            get_pod_reallocation_spec_response = yp_lite_reallocation_service_stub.get_pod_reallocation_spec(
                get_pod_reallocation_spec_request)
        current_reallocation_id = get_pod_reallocation_spec_response.spec.id
        # Use current reallocation ID as 'previous_reallocation_id'.
        start_pod_reallocation_request.previous_reallocation_id = current_reallocation_id

        # Launch reallocation.
        with stat_server.yp_lite_timing():
            start_pod_reallocation_response = yp_lite_reallocation_service_stub.start_pod_reallocation(
                start_pod_reallocation_request)

        self.logger.info("Successfully started reallocation, ID %s" % start_pod_reallocation_response.reallocation_id)

    def _allow_reallocation(self, ts: int, config_interface: ConfigInterface) -> bool:
        target_ts = ts - 60

        client = nanny_client.NannyClient(self.secrets_map["NANNY_OAUTH_TOKEN"])
        with stat_server.nanny_timing():
            last_change_time = client.get_change_time(self.name)
        if last_change_time > target_ts - 120:
            return False

        clickhouse_client = ClickhouseClient(config_interface)

        reallocation_history = ReallocationHistory(nanny_service_name=self.name, config_interface=config_interface,
                                                   clickhouse_client=clickhouse_client, end_ts=target_ts)

        if reallocation_history.if_session_in_progress():
            self.logger.info("Reallocation not allowed: another reallocation session is in progress")
            return False

        redeploy_configured = config_interface.get_tentacles_group_redeployment_settings(self.name)
        if not redeploy_configured:
            self.logger.info(("Reallocation allowed: no another reallocation session is in progress, "
                              "redeployment is not configured."))
            return True

        redeployment_history = RedeploymentHistory(nanny_service_name=self.name, config_interface=config_interface,
                                                   clickhouse_client=clickhouse_client, end_ts=target_ts)

        if redeployment_history.if_session_in_progress():
            self.logger.info("Reallocation not allowed: redeployment configured and is in progress")
            return False

        # Check if there are complete sessions in timeline.
        if reallocation_history.get_last_complete_session_borders() == (None, None) and \
           redeployment_history.get_last_complete_session_borders() == (None, None):
            raise RerollHistoryException(("No complete sessions present in Nanny state history window, can not "
                                          "determine whether it is time to reallocate or redeploy."))

        # Launch reallocation and redeploy one after another.
        reallocation_idle_time = reallocation_history.get_current_period_duration()
        redeployment_idle_time = redeployment_history.get_current_period_duration()
        self.logger.debug((f"reallocation_idle_time={reallocation_idle_time}, "
                           f"redeployment_idle_time={redeployment_idle_time}"))
        if reallocation_idle_time > redeployment_idle_time:
            self.logger.info("Reallocation allowed: redeployment configured and just finished.")
            return True
        self.logger.info(("Reallocation not allowed: redeployment configured, another reallocation session "
                          "just finished."))
        return False

    def _get_nanny_snapshot_id(self) -> str:
        snapshot_labels = self._snapshot_manager.find_labels(
            harvester_type='nanny_state_dumper',
            harvester_name=self.name,
            meta_query={'$ne': None},
        )

        if not snapshot_labels:
            raise Exception("No snapshot available")

        snapshot_data = self._snapshot_manager.read_snapshot(label=snapshot_labels[0]).data

        return snapshot_data['current_active']['snapshot_id']
