import dataclasses
import json
import typing

from infra.nanny.nanny_services_rest.nanny_services_rest.client import ServiceRepoClient
from infra.nanny.nanny_services_rest.nanny_services_rest.errors import ModificationConflictError

from infra.rtc_sla_tentacles.backend.lib.clickhouse.client import ClickhouseClient
from infra.rtc_sla_tentacles.backend.lib.config.interface import ConfigInterface
from infra.rtc_sla_tentacles.backend.lib.harvesters.base import Harvester
from infra.rtc_sla_tentacles.backend.lib.harvesters_snapshots.snapshots import HarvesterSnapshot
from infra.rtc_sla_tentacles.backend.lib.reroll_history.history import ReallocationHistory, RedeploymentHistory, \
    RerollHistoryException
from infra.rtc_sla_tentacles.backend.lib.tentacle_agent import const as agent_const
from infra.rtc_sla_tentacles.backend.lib.yp_lite.pods_manager import YpLitePodsManager
from infra.rtc_sla_tentacles.backend.lib.nanny import nanny_client
from infra.rtc_sla_tentacles.backend.lib.funccall_stats_server import server as stat_server


TAttributes = typing.Dict


@dataclasses.dataclass
class AgentResource:
    rbtorrentid: str
    tar: bool
    resource_name: str


# noinspection PyProtectedMember
class YpLiteSwitcher(Harvester):
    harvester_type = 'yp_lite_switcher'

    secrets_map = {
        "NANNY_OAUTH_TOKEN": None
    }

    def extract(self, ts: int) -> typing.Optional[AgentResource]:
        self.arguments = self.config_interface.get_tentacles_group_redeployment_settings(self.name)

        if not self._allow_redeploy(ts):
            return None

        nanny_rest_client_config_kwargs = self.config_interface.get_nanny_rest_client_config()
        nanny_service_repo_client = ServiceRepoClient(**nanny_rest_client_config_kwargs)

        pods_manager = None
        if self.arguments.get("update_nanny_instances_with_allocated_pods") is not None:
            pods_manager = self._get_pods_manager(self.name, self.config_interface)
            self.logger.info("Loading YP pods manager from ClickHouse...")
            with pods_manager as pods_manager_with_clients:
                pods_manager_with_clients.load_pods_from_clickhouse(ts_to_wait_for=ts)

        # SPI-9888 , https://wiki.yandex-team.ru/runtime-cloud/nanny/service-repo-api/#modification-conflicts
        num_tries = 2
        changed = False
        agent_resource = self._get_most_fresh_resource()
        while not changed and num_tries:
            try:
                request = self._get_runtime_attrs_change_request(
                    agent_resource, nanny_service_repo_client, pods_manager
                )
                with stat_server.nanny_timing():
                    nanny_service_repo_client.put_runtime_attrs(s_id=self.name, content=request)
                changed = True
            except ModificationConflictError:
                num_tries -= 1
                continue
        if not changed:
            raise ModificationConflictError("Got ModificationConflictError exception %d times, surrender" % num_tries)

        return agent_resource

    def transform(self, ts: int, raw_data: typing.Optional[AgentResource]) -> typing.Tuple[dict, dict]:
        if not raw_data:
            data = {'resource_id': None}
        else:
            data = {'resource_id': raw_data.rbtorrentid}
        meta = {'success': True}
        return meta, data

    def _allow_redeploy(self, ts: int) -> bool:
        target_ts = ts - 60

        client = nanny_client.NannyClient(self.secrets_map["NANNY_OAUTH_TOKEN"])
        with stat_server.nanny_timing():
            last_change_time = client.get_change_time(self.name)
        if last_change_time > target_ts - 120:
            return False

        clickhouse_client = ClickhouseClient(self.config_interface)

        redeployment_history = RedeploymentHistory(
            nanny_service_name=self.name,
            config_interface=self.config_interface,
            clickhouse_client=clickhouse_client,
            end_ts=target_ts,
        )

        if redeployment_history.if_session_in_progress():
            self.logger.info("Redeploy not allowed: another redeploy session is in progress")
            return False

        reallocation_configured = self.config_interface.get_tentacles_group_reallocation_settings(self.name)
        if not reallocation_configured:
            self.logger.info(("Redeploy allowed: no another redeploy session is in progress, "
                              "reallocation is not configured."))
            return True

        reallocation_history = ReallocationHistory(
            nanny_service_name=self.name,
            config_interface=self.config_interface,
            clickhouse_client=clickhouse_client,
            end_ts=target_ts,
        )

        if reallocation_history.if_session_in_progress():
            self.logger.info("Redeploy not allowed: reallocation configured and is in progress")
            return False

        # Check if there are complete sessions in timeline.
        if redeployment_history.get_last_complete_session_borders() == (None, None) and \
           reallocation_history.get_last_complete_session_borders() == (None, None):
            raise RerollHistoryException(("No complete sessions present in Nanny state history window, can not "
                                          "determine whether it is time to reallocate or redeploy."))

        # Launch redeploy and reallocation one after another.
        redeployment_idle_time = redeployment_history.get_current_period_duration()
        reallocation_idle_time = reallocation_history.get_current_period_duration()
        self.logger.debug((f"redeployment_idle_time={redeployment_idle_time}, "
                           f"reallocation_idle_time={reallocation_idle_time}"))
        if redeployment_idle_time > reallocation_idle_time:
            self.logger.info("Redeploy allowed: reallocation configured and just finished.")
            return True
        self.logger.info(("Redeploy not allowed: reallocation configured, another redeploy session "
                          "just finished."))
        return False

    def _get_most_fresh_resource(self) -> AgentResource:
        resources = self._snapshot_manager.find_labels(
            harvester_type='resource_maker',
            harvester_name=self.arguments['resource_maker'],
            meta_query={'success': True},
        )
        if not resources:
            raise Exception("No 'resource_maker' successful runs found")

        resource_label = resources[0]
        resource_snapshot = self._snapshot_manager.read_snapshot(resource_label)
        return self._get_agent_resource(resource_snapshot)

    @staticmethod
    def _get_agent_resource(resource_snapshot: HarvesterSnapshot) -> AgentResource:
        return AgentResource(
            resource_snapshot.data['resource_id'],
            tar=False,
            resource_name=agent_const.AGENT_RESOURCE_NAME,
        )

    def _get_pods_manager(self, nanny_service_name: str, config_interface: ConfigInterface) -> YpLitePodsManager:
        yp_cluster = config_interface.tentacles_groups_config.get_option(nanny_service_name, "yp_cluster")
        restrict_nodes_to_specified_in_podset = config_interface.tentacles_groups_config.get_option(
            nanny_service_name, "restrict_nodes_to_specified_in_podset")
        return YpLitePodsManager(
            nanny_service_name=nanny_service_name,
            yp_cluster=yp_cluster,
            logger=self.logger,
            restrict_nodes_to_specified_in_podset=restrict_nodes_to_specified_in_podset,
            config_interface=config_interface,
            init_clickhouse_client=True,
        )

    def _get_runtime_attrs_change_request(
            self, agent_resource: AgentResource, nanny_service_repo_client: ServiceRepoClient,
            pods_manager: YpLitePodsManager) -> dict:
        # Read current Nanny runtime attributes.
        with stat_server.nanny_timing():
            active_runtime_attrs = nanny_service_repo_client.get_runtime_attrs(self.name)
        # Insert rbtorrentid.
        active_runtime_attrs = self._insert_rbtorrentid_to_nanny_runtime_attributes(
            active_runtime_attrs, agent_resource
        )
        self.logger.info(f"Inserted {agent_resource!r} into Nanny files configuration")
        comment = f"Bump {agent_resource.resource_name}, {agent_resource.rbtorrentid}"

        if pods_manager is not None:
            # Replace instances with allocated pods.
            active_runtime_attrs = pods_manager.insert_allocated_pods_to_nanny_runtime_attributes(active_runtime_attrs)
            num_allocated_pods = len(active_runtime_attrs["content"]["instances"]["yp_pod_ids"]["pods"])
            self.logger.info(f"Inserted {num_allocated_pods!r} allocated pods into Nanny instances configuration")
            comment = ", ".join([comment, f"updated pods list with {num_allocated_pods} allocated pods"])

        return {
            "snapshot_id": active_runtime_attrs["_id"],
            "content": active_runtime_attrs["content"],
            "comment": comment,
            "meta_info": {
                "scheduling_config": {
                    "scheduling_priority": "CRITICAL",
                }
            }
        }

    @staticmethod
    def _insert_rbtorrentid_to_nanny_runtime_attributes(
        _attrs: TAttributes, agent_resource: AgentResource
    ) -> TAttributes:
        try:
            _updated = False
            for url_file in _attrs["content"]["resources"]["url_files"]:
                if url_file["local_path"] == agent_resource.resource_name:
                    url_file["url"] = agent_resource.rbtorrentid
                    _updated = True
                    break
            if not _updated:
                _attrs["content"]["resources"]["url_files"].append({
                    "local_path": agent_resource.resource_name,
                    "url": agent_resource.rbtorrentid,
                })
        except (KeyError, AttributeError):
            raise Exception("Got malformed snapshot data dict from Nanny:\n%s" % json.dumps(_attrs, indent=4))
        return _attrs
