import json
import urllib.request

from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager import config, slo_checks
from infra.rtc_sla_tentacles.backend.lib.config.interface import ConfigInterface


class Generator:
    # NOTE(rocco66): inspired by juggler/playbooks/self-monitoring/yasm.py

    def __init__(self, prefix, ctype):
        self._prefix = prefix
        self._ctype = ctype
        self.alerts = []

    @staticmethod
    def make_limits(crit_thresh, warn_thresh):
        if crit_thresh > warn_thresh:
            return [warn_thresh, crit_thresh], [crit_thresh, None]
        else:
            return [crit_thresh, warn_thresh], [None, crit_thresh]

    def make_juggler_check(self, host, service, stable=90, critical=600, tags=None):
        tags = tags or []
        return {
            "host": host,
            "service": service,
            "refresh_time": 5,
            "namespace": config.BACKEND_JUGGLER_NAMESPACE,
            "aggregator": "logic_or",
            "flaps": {"stable": stable, "critical": critical},
            "tags": [self._ctype] + tags,
        }

    def make_alert(self, name, signal, itype, crit_thresh, warn_thresh,
                   tags=None, alert_kwargs=None, percent_thresholds=False,
                   juggler_tags=None):
        full_name = "{0}.{1}".format(self._prefix, name)
        tags_ = {
            "itype": [itype]
        }
        if tags:
            tags_.update(tags)
        if self._ctype:
            tags_["ctype"] = [self._ctype]
        warn_limit, crit_limit = (warn_thresh, crit_thresh) if percent_thresholds else self.make_limits(crit_thresh,
                                                                                                        warn_thresh)
        warn_literal, crit_literal = ("warn_perc", "crit_perc") if percent_thresholds else ("warn", "crit")
        alert = {
            crit_literal: crit_limit,
            warn_literal: warn_limit,
            "mgroups": ["ASEARCH"],
            "name": full_name,
            "abc": config.TENTACLES_ABC_SERVICE,
            "signal": signal,
            "tags": tags_
        }
        if alert_kwargs is not None:
            alert.update(alert_kwargs)
        alert["juggler_check"] = self.make_juggler_check(
            host=f"tentacles.{self._ctype}",
            service=name,
            tags=juggler_tags,
        )
        self.alerts.append(alert)


def _generate_alerts_for_warden(gen, config_interface):
    # NOTE(rocco66): TENTACLES-238
    for allocation_zone_id, allocation_zone_config in config_interface.get_allocation_zones().items():
        availability_config = config_interface.get_tentacles_group_availability_settings(allocation_zone_id)
        if warden_slo_limit := availability_config.get("monitoring", {}).get("warden_slo_limit"):
            gen.make_alert(
                f"warden.{allocation_zone_id}.availability",
                f"unistat-az__{allocation_zone_id}__percent_of_available_tentacles_axxv",
                itype=config.API_ITYPE,
                crit_thresh=warden_slo_limit,
                warn_thresh=warden_slo_limit,
                juggler_tags=slo_checks.get_geo_tags(allocation_zone_config, allocation_zone_id)
            )


def _generate_alerts_for_yp_lite_scheduling(gen, config_interface):
    # NOTE(rocco66): see TENTACLES-281 for details
    zones = config_interface.get_allocation_zones(yp_daemonsets_only=True)
    for allocation_zone_id, allocation_zone_config in zones.items():
        limit = config_interface.tentacles_groups_config.get_scheduling_errors_limit(allocation_zone_id)
        gen.make_alert(
            f"yp_lite.{allocation_zone_id}.scheduling_perc",
            (
                f"na(perc("
                f"unistat-podsmanager__{allocation_zone_id}__nodes_up_scheduling_errors__axxv,"
                f"unistat-podsmanager__{allocation_zone_id}__nodes_up_seen__axxv))"
            ),
            itype=config.API_ITYPE,
            crit_thresh=limit,
            warn_thresh=limit - 0.01,
            juggler_tags=slo_checks.get_geo_tags(allocation_zone_config, allocation_zone_id) + [
                "yp_daemonset_scheduling",
                "slo",
                f"az_{allocation_zone_id}",
                config.CHAT_NOTIFICATION_TAG,
            ],
            alert_kwargs={
                "value_modify": {
                    "type": "aver",
                    "window": 600,
                },
            },
        )


def generate_alerts(prefix, ctype, file_config):
    gen = Generator(prefix, ctype=ctype)

    gen.make_alert(
        "free_workers",
        "unistat-worker_free_count_ammv",
        itype=config.WORKER_ITYPE,
        crit_thresh=13,
        warn_thresh=15,
        alert_kwargs={
            "value_modify": {
                "type": "aver",
                "window": 60,
            },
        },
        juggler_tags=[config.CHAT_NOTIFICATION_TAG],
    )

    config_interface = ConfigInterface(full_config=file_config)
    _generate_alerts_for_warden(gen, config_interface)
    _generate_alerts_for_yp_lite_scheduling(gen, config_interface)

    return gen.alerts


def replace(prefix, alerts):
    request = urllib.request.Request(
        url="https://yasm.yandex-team.ru/srvambry/alerts/replace",
        data=json.dumps({
            "prefix": prefix,
            "alerts": alerts
        }).encode("utf8"),
        headers={
            "Content-Type": "application/json",
            "User-Agent": "tentacles-self-monitoring-generator"
        }
    )
    return urllib.request.urlopen(request, timeout=15)
