import dataclasses
import datetime
import typing

import humanize

from juggler_sdk import Check, FlapOptions, Child

from infra.rtc_sla_tentacles.backend.lib.config.interface import ConfigInterface
from infra.rtc_sla_tentacles.backend.lib.config.harvesters_config import HarvestersConfig
from infra.rtc_sla_tentacles.backend.lib.config.tentacles_groups_config import TentaclesGroupsConfig
from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager.config import BACKEND_JUGGLER_NAMESPACE, \
    CHAT_NOTIFICATION_TAG, STAGE_CONFIG
from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager.util import add_event_prefix
from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager import slo_checks
from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager import util as checks_util


RESOUCE_MAKING_PROCESS_SERVICE = "resource_making_process"
JUGGLER_HOST = "tentacles_backend_harvesters_snapshots_freshness"

YP_DEPS_HARVESTERS = {
    "hq",
    "yp_lite_allocation",
    "yp_lite_pods_tracker",
    "yp_lite_pod_count_tuner",
    "yp_lite_switcher",
    "yp_lite_reallocator",
}
YP_REALEASE_MAX_DOWNTIME = 10 * 60


@dataclasses.dataclass
class _HarvesterDescription:
    type: str
    interval: int
    several_harvesters: bool
    name: str = None
    geo_tags: typing.List[str] = None


def make_juggler_host(stage):
    if stage == "production":
        return JUGGLER_HOST
    else:
        return f"{JUGGLER_HOST}_{stage}"


def make_juggler_service(harvester_type, harvester_name, several_harvesters):
    if several_harvesters:
        return f"{harvester_type}__{harvester_name}"
    else:
        return harvester_type


def get_freshness_checks(stage: str, file_config: dict) -> typing.List[Check]:
    return [
        _get_harvester_freshness_check(harvester_dsc=check_config, stage=stage, file_config=file_config)
        for check_config in _get_checks_configs(file_config)
    ] + [_resource_check(stage)]


def _get_tags(stage):
    return ["harvesters_snapshots_freshness", CHAT_NOTIFICATION_TAG, stage]


def _resource_check(stage: str) -> Check:
    stage_params = STAGE_CONFIG[stage]
    mark = stage_params["checks_sets_marks"]["harvesters_snapshots_freshness"]
    making_interval = 300
    service = add_event_prefix(stage, RESOUCE_MAKING_PROCESS_SERVICE)
    return Check(
        namespace=BACKEND_JUGGLER_NAMESPACE,
        host=f"tentacles.{stage}",
        service=service,
        tags=_get_tags(stage) + [f"a_ctype_{stage}"],
        children=[Child(
            host=stage_params["backend_common_checks"]["worker"]["juggler_hostname"],
            service=service,
            group_type="NANNY",
        )],
        aggregator="more_than_limit_is_crit",
        aggregator_kwargs={
            "nodata_mode": "force_crit",
            "percent": 51,
        },
        ttl=making_interval * 2.5,
        flaps_config=FlapOptions(stable=making_interval * 2.5, critical=making_interval * 15),
        mark=mark
    )


def _get_harvester_freshness_check(harvester_dsc: _HarvesterDescription, stage: str, file_config: dict) -> Check:
    host = make_juggler_host(stage)
    service = make_juggler_service(harvester_dsc.type, harvester_dsc.name, harvester_dsc.several_harvesters)
    tags = _get_tags(stage)
    if geo_tags := harvester_dsc.geo_tags:
        tags.extend(geo_tags)

    ttl = harvester_dsc.interval * 4
    if harvester_dsc.type in YP_DEPS_HARVESTERS:
        ttl = max(ttl, YP_REALEASE_MAX_DOWNTIME)

    if harvester_dsc.several_harvesters:
        harvester_name_repr = f"{harvester_dsc.type}/{harvester_dsc.name}"
    else:
        harvester_name_repr = harvester_dsc.type
    nodata_desc = (
        f"Harvester {harvester_name_repr} has not been successfully "
        f"executed within the last {humanize.naturaldelta(datetime.timedelta(seconds=ttl))}"
    )
    if harvester_dsc.several_harvesters:
        allocation_zone_id = harvester_dsc.name
        config_interface = ConfigInterface(full_config=file_config)
        nodata_desc += checks_util.get_allocation_zone_description_part(config_interface, allocation_zone_id)

    result = Check(
        namespace=BACKEND_JUGGLER_NAMESPACE,
        host=host,
        service=service,
        tags=tags,
        children=[Child(add_event_prefix(stage, host), add_event_prefix(stage, service))],
        aggregator="logic_or",
        aggregator_kwargs={
            "nodata_desc": nodata_desc,
        },
        refresh_time=60,
        ttl=ttl,
        mark=STAGE_CONFIG[stage]["checks_sets_marks"]["harvesters_snapshots_freshness"]
    )
    return result


def _get_harvester_interval(harvester_config):
    return harvester_config["common_settings"]["update_interval_sec"]


def _get_checks_configs(file_config: dict) -> typing.List[_HarvesterDescription]:
    harvesters_config_dict = HarvestersConfig(full_config=file_config).to_dict()
    config = []
    for item in _get_monitored_types(harvesters_config_dict):
        config.append(item)
    tentacles_groups_config = TentaclesGroupsConfig(full_config=file_config).to_dict()
    for item in _get_monitored_types_names(harvesters_config_dict, tentacles_groups_config):
        config.append(item)
    return config


def _get_monitored_types(harvesters_config_dict: dict) -> typing.Iterator[_HarvesterDescription]:
    for type_name in harvesters_config_dict["harvesters"]:
        harvester_config = harvesters_config_dict["harvesters"][type_name]
        harvester_instances = harvester_config.get("arguments")
        if len(harvester_instances) == 1:
            yield _HarvesterDescription(
                type_name,
                _get_harvester_interval(harvester_config),
                several_harvesters=False,
            )


def _get_monitored_types_names(harvesters_config_dict: dict,
                               tentacles_groups_config: dict) -> typing.Iterator[_HarvesterDescription]:
    for type_name in harvesters_config_dict["harvesters"]:
        harvester_config = harvesters_config_dict["harvesters"][type_name]
        harvester_instances = harvester_config.get("arguments")
        if harvester_instances and len(harvester_instances) > 1:
            for harvester_name in harvester_instances:
                geo_tags = slo_checks.get_geo_tags(tentacles_groups_config[harvester_name], harvester_name)
                yield _HarvesterDescription(
                    type_name,
                    _get_harvester_interval(harvester_config),
                    several_harvesters=True,
                    name=harvester_name,
                    geo_tags=geo_tags
                )
