import typing

import juggler_sdk

from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager import config as checks_config
from infra.rtc_sla_tentacles.backend.lib.config.interface import ConfigInterface
from infra.rtc_sla_tentacles.backend.lib.metrics import checks as metrics_checks, types
from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager import util as checks_util

FORCED_GEO_TAGS = {
    "yp_daemonset_xdc_rtc_sla_tentacles_production": ["sas", "man", "iva", "myt"],
    "yp_daemonset_man_pre_rtc_sla_tentacles_production": ["sas", "man"],
}


def _get_flap_options(config_interface, configured_check: metrics_checks.ConfiguredCheck) -> juggler_sdk.FlapOptions:
    stable_time = 4 * 60
    # See JUGGLERSUPPORT-1774 , may be removed after JUGGLER-2145 .
    is_availability_slo = configured_check.slo_type == types.SloType.availability  # noqa E721
    is_reallocation_zone = (
        configured_check.allocation_zone_id in config_interface.get_tentacles_groups_with_reallocation()
    )
    if is_availability_slo and is_reallocation_zone:
        stable_time = 10 * 60
    return juggler_sdk.FlapOptions(stable_time, stable_time * 5)


def get_geo_tags(allocation_zone_config, allocation_zone_id: str):
    tags = FORCED_GEO_TAGS.get(allocation_zone_id)
    if not tags:
        location = allocation_zone_config["location"]
        # TENTACLES-69
        if location == "msk":
            tags = ["myt", "iva"]
        else:
            tags = [location]
    return [f"a_geo_{t}" for t in tags]


def get_slo_checks(file_config, stage: str) -> typing.Iterable[juggler_sdk.Check]:
    config_interface = ConfigInterface(full_config=file_config)
    stage_params = checks_config.STAGE_CONFIG[stage]
    for allocation_zone_id in config_interface.get_allocation_zones():
        all_checks = list(metrics_checks.get_fast_configured_checks(config_interface, allocation_zone_id))
        # Disabled via TENTACLES-368 .
        # if config_utils.is_daemonset_location(allocation_zone_id):
        #     all_checks.append(metrics_checks.get_yp_unused_nodes_slow_configured_check(config_interface, allocation_zone_id))
        for configured_check in all_checks:
            stage = config_interface.get_env_name()
            tags = [
                "slo",
                f"slo_type_{configured_check.slo_type}",
                f"az_{configured_check.allocation_zone_id}",
                stage
            ]
            allocation_zone_config = config_interface.get_allocation_zone_config(allocation_zone_id)
            tags.extend(get_geo_tags(allocation_zone_config, allocation_zone_id))
            if not (
                configured_check.host.endswith("_flow") or
                configured_check.slo_type == types.SloType.unused_nodes or  # noqa E721
                configured_check.slo_type == types.SloType.redeployed_on_time  # TENTACLES-391
            ):
                tags.append(checks_config.CHAT_NOTIFICATION_TAG)

            yield juggler_sdk.Check(
                namespace=checks_config.BACKEND_JUGGLER_NAMESPACE,
                host=configured_check.host,
                service=configured_check.service,
                tags=tags,
                aggregator="logic_or",
                aggregator_kwargs={},
                refresh_time=60,
                ttl=configured_check.ttl,
                children=[juggler_sdk.Child(
                    host=checks_util.add_event_prefix(stage, configured_check.host),
                    service=checks_util.add_event_prefix(stage, configured_check.service),
                )],
                flaps_config=_get_flap_options(config_interface, configured_check),
                mark=stage_params["checks_sets_marks"]["slo"],
            )
