import typing

from infra.rtc_sla_tentacles.backend.lib.metrics.types import SloType, ConfiguredCheck


# TODO(rocco66): rename old check's service to allocation_zone
ALLOCATION_ZONE_TO_SERVICE = {
    "rtc_sla_tentacles_production_sas": "sas",
    "rtc_sla_tentacles_production_vla": "vla",
    "rtc_sla_tentacles_production_man": "man",
    "rtc_sla_tentacles_production_msk": "msk",
    "yp_rtc_sla_tentacles_production_sas": "yp_sas",
    "yp_rtc_sla_tentacles_production_vla": "yp_vla",
    "yp_rtc_sla_tentacles_production_man": "yp_man",
    "yp_rtc_sla_tentacles_production_man_pre": "yp_man_pre",
}


def _make_check(allocation_zone_id, slo_type,  config_interface, limit=0, additional_postfix="", ttl=5*60):
    stage = config_interface.get_env_name()
    if stage == "production":
        host_postfix = ""
    elif stage == "testing":
        host_postfix = "_testing"
    else:
        raise RuntimeError(f"unknown stage {stage}")
    if additional_postfix:
        host_postfix += additional_postfix

    return ConfiguredCheck(
        allocation_zone_id=allocation_zone_id,
        host=f"rtc_sla_{slo_type}{host_postfix}",
        service=ALLOCATION_ZONE_TO_SERVICE.get(allocation_zone_id, allocation_zone_id),
        slo_type=slo_type,
        limit=limit,
        ttl=ttl,
    )


def get_fast_configured_checks(config_interface, allocation_zone_id) -> typing.Iterable[ConfiguredCheck]:
    stage = config_interface.get_env_name()
    if stage == "production" and allocation_zone_id == "yp_rtc_sla_tentacles_production_man_pre":
        # NOTE(rocco66): this allocation zone managed by testing and might be broken
        return

    allocation_zones_config = config_interface.tentacles_groups_config
    yield _make_check(
        allocation_zone_id,
        SloType.availability,
        config_interface,
        allocation_zones_config.get_availability_limit(allocation_zone_id),
    )
    yield _make_check(
        allocation_zone_id,
        SloType.redeployed_on_time,
        config_interface,
        allocation_zones_config.get_redeployment_limit(allocation_zone_id),
    )

    if allocation_zones_config.is_reallocation_enable(allocation_zone_id):
        yield _make_check(allocation_zone_id, SloType.reallocation, config_interface)

    if flow_availability_limit := allocation_zones_config.get_flow_availability_limit(allocation_zone_id):
        yield _make_check(
            allocation_zone_id,
            SloType.availability,
            config_interface,
            flow_availability_limit,
            additional_postfix="_flow"
        )


def get_yp_unused_nodes_slow_configured_check(config_interface, allocation_zone_id):
    harvester_config = config_interface.get_harvester_config("yp_unused_nodes_monitoring")
    harvester_interval = harvester_config["common_settings"]["update_interval_sec"]
    return _make_check(
        allocation_zone_id,
        SloType.unused_nodes,
        config_interface,
        ttl=harvester_interval * 4,
    )
