import logging
import time
import typing

from dataclasses import dataclass

from infra.rtc_sla_tentacles.backend.lib.clickhouse.client import ClickhouseClient
from infra.rtc_sla_tentacles.backend.lib.clickhouse.database import NannyState
from infra.rtc_sla_tentacles.backend.lib.config.interface import ConfigInterface
from infra.rtc_sla_tentacles.backend.lib.reroll_history.timeline import ReallocationTimeline, RedeploymentTimeline,\
    TimelineFrame


class RerollHistoryException(Exception):
    pass


@dataclass
class RerollHistorySession:
    start_ts: int
    end_ts: int
    frames: typing.List[TimelineFrame]
    taskgroup_ids: typing.List[str]
    snapshot_ids: typing.List[str]
    in_progress_duration: int
    cooldown_duration: int

    @property
    def end_of_active_phase(self):
        return self.start_ts + self.in_progress_duration


class BaseRerollHistory:
    logger = logging.getLogger("reroll_history")

    _default_load_window_hours = 24
    _in_progress_statuses = []
    _ts_step = 60

    def __init__(self,
                 nanny_service_name: str,
                 config_interface: ConfigInterface = None,
                 clickhouse_client: ClickhouseClient = None,
                 start_ts: int = None,
                 end_ts: int = None,
                 wait_for_end_ts: bool = False):
        """
        * Reads Nanny service name history from ClickHouse
        * Creates Timeline object from history
        * Marks timeline frames with periods when reroll
          session is active or idle, with respect to cooldowns
          set in config.
        * Provides several search interfaces to stored Timeline.
        """
        if end_ts is None:
            end_ts = int((time.time() // 60) * 60)
        self._end_ts = end_ts

        if start_ts is None:
            start_ts = end_ts - self._default_load_window_hours * 60 * 60
        self._start_ts = start_ts

        self.logger = type(self).logger.getChild(nanny_service_name)
        self._nanny_service_name = nanny_service_name
        self._config_interface = config_interface or ConfigInterface()

        if not clickhouse_client:
            clickhouse_client = ClickhouseClient(self._config_interface)
        self._clickhouse_client = clickhouse_client

        self._timeline = self._read_timeline(self._start_ts, self._end_ts, wait_for_end_ts)
        self._mark_sessions()

    def __str__(self):
        return f"{self.__class__.__name__}({self._nanny_service_name}, start={self._start_ts}, end={self._end_ts})"

    def if_session_in_progress(self) -> bool:
        return self._timeline[-1].props["session_in_progress"]

    def if_cooldown_in_progress(self) -> bool:
        return self._timeline[-1].props["cooldown_in_progress"]

    def get_current_period_duration(self) -> int:
        return self._timeline[-1].props["period_end_ts"] - self._timeline[-1].props["period_start_ts"]

    def get_current_period_borders(self) -> (int, int):
        return self._timeline[-1].props["period_start_ts"], self._timeline[-1].props["period_end_ts"]

    def get_last_complete_session_borders(self) -> (typing.Optional[int], typing.Optional[int]):
        previous_frame = self._timeline[-1]
        for _i in range(len(self._timeline) - 2, -1, -1):
            current_frame = self._timeline[_i]
            if previous_frame.props["period_start_ts"] != current_frame.props["period_start_ts"] and \
               current_frame.props["session_in_progress"]:
                return current_frame.props["period_start_ts"], current_frame.props["period_end_ts"]
        return None, None

    def get_last_complete_session(self) -> typing.Optional[RerollHistorySession]:
        start_ts, end_ts = self.get_last_complete_session_borders()
        if start_ts is None and end_ts is None:
            return None
        return self._get_session(start_ts, end_ts)

    def get_current_session(self) -> typing.Optional[RerollHistorySession]:
        if not self.if_session_in_progress():
            return None
        start_ts, end_ts = self.get_current_period_borders()
        return self._get_session(start_ts, end_ts)

    def get_last_success_active_session(self):
        if self.if_session_in_progress() and self.if_cooldown_in_progress():
            return self.get_current_session()
        else:
            return self.get_last_complete_session()

    def get_current_active_session(self):
        if self.if_session_in_progress() and not self.if_cooldown_in_progress():
            return self.get_current_session()

    def get_complete_sessions(self, limit=3) -> typing.List[RerollHistorySession]:
        """ Returns list of sessions sorted from newest to oldest. """
        result = []
        previous_frame = self._timeline[-1]
        counter = 0
        for _i in range(len(self._timeline) - 2, -1, -1):
            current_frame = self._timeline[_i]
            if current_frame.props["period_start_ts"] != previous_frame.props["period_start_ts"] and \
                    current_frame.props["session_in_progress"]:
                result.append(self._get_session(
                    current_frame.props["period_start_ts"], current_frame.props["period_end_ts"]))
                counter += 1
                if limit and counter == limit:
                    break
            previous_frame = current_frame
        return result

    def _get_cooldown_duration_sec(self):
        raise NotImplementedError

    def _get_session(self, start_ts: int, end_ts: int) -> RerollHistorySession:
        frames = self._timeline.get_frames(start_ts, end_ts)
        taskgroup_ids = []
        snapshot_ids = []
        in_progress_duration = 0
        cooldown_duration = 0

        for frame in frames:
            if frame.props["cooldown_in_progress"]:
                cooldown_duration += self._ts_step
            if frame.state in self._in_progress_statuses:
                in_progress_duration += self._ts_step
            if frame.taskgroup_id and frame.taskgroup_id not in taskgroup_ids:
                taskgroup_ids.append(frame.taskgroup_id)
            if frame.latest_snapshot_id and frame.latest_snapshot_id not in snapshot_ids:
                snapshot_ids.append(frame.latest_snapshot_id)

        return RerollHistorySession(
            start_ts, end_ts, frames, taskgroup_ids, snapshot_ids, in_progress_duration, cooldown_duration)

    def _mark_frame_with_session(self, index: int, period_start_ts: int, if_session_in_progress: bool):
        self._timeline[index].props["period_start_ts"] = period_start_ts
        self._timeline[index].props["session_in_progress"] = if_session_in_progress

    # noinspection PyUnusedLocal
    def _mark_sessions(self):
        """
        Reroll session is a sequence of timeframes, which statuses are
        "in progress statuses" (e.g. ["PREPARING", "UPDATING"]), followed
        by a cooldown period.
        This method marks each frame in timeline with the set of properties:
        * session_in_progress - Shows if this frame is a part of some reroll
              session (True), or a part of period without reroll (False).
        * cooldown_in_progress - Shows if this frame belongs to a session,
              and is treated as cooldown - not in "in progress statuses".
              Cooldown time is a part of a session.
        * period_start_ts, period_end_ts - Timestamp that define borders
              of the sessions and periods of idleness between them.

        Empty frames - without status (missing data) - are treated well.
        """
        cooldown_duration = self._get_cooldown_duration_sec()
        period_start_ts = self._start_ts

        # Mark frames without data at the start of timeline with
        # value of the first non-empty frame.
        oldest_non_empty_frame_index = self._timeline.get_oldest_non_empty_frame_index()
        oldest_non_empty_frame = self._timeline[oldest_non_empty_frame_index]
        if_session_in_progress = oldest_non_empty_frame.state in self._in_progress_statuses
        for _i in range(0, oldest_non_empty_frame_index + 1):
            self._mark_frame_with_session(_i, period_start_ts, if_session_in_progress)

        # Check if oldest non-empty timeframe is the last one.
        if oldest_non_empty_frame_index == len(self._timeline) - 1:
            for _i in range(0, oldest_non_empty_frame_index + 1):
                self._timeline[_i].props["period_end_ts"] = self._start_ts
            return

        cooldown_end_ts = 0

        for _i in range(oldest_non_empty_frame_index + 1, len(self._timeline)):
            current_frame = self._timeline[_i]
            previous_frame = self._timeline[_i - 1]

            # If current frame is empty - prolong previous session status.
            if current_frame.state is None:
                self._mark_frame_with_session(_i, period_start_ts, previous_frame.props["session_in_progress"])

            # If previous frame is empty - use most recent non-empty frame as previous.
            if previous_frame.state is None:
                for _k in range(_i - 2, -1, -1):
                    if self._timeline[_k].state is not None:
                        previous_frame = self._timeline[_k]

            current_frame_in_progress = current_frame.state in self._in_progress_statuses
            previous_frame_in_progress = previous_frame.state in self._in_progress_statuses

            cooldown_active = current_frame.ts < cooldown_end_ts

            if current_frame_in_progress:  # Current frame is "in progress".

                if cooldown_active:  # Active cooldown.
                    # This is not expected to happen. Reset active cooldown, start new session.
                    cooldown_end_ts = 0
                    period_start_ts = current_frame.ts
                    self._mark_frame_with_session(_i, period_start_ts, True)

                else:  # No active cooldown.
                    if previous_frame_in_progress:  # Previous frame is "in progress".
                        # Prolong current session.
                        self._mark_frame_with_session(_i, period_start_ts, True)
                    else:  # Previous frame is not "in progress".
                        # Start new session.
                        period_start_ts = current_frame.ts
                        self._mark_frame_with_session(_i, period_start_ts, True)

            else:  # Current frame is not "in progress".

                if cooldown_active:  # Active cooldown.
                    # Prolong session.
                    self._mark_frame_with_session(_i, period_start_ts, True)

                else:  # No active cooldown.
                    if previous_frame_in_progress:  # Previous frame is "in progress".
                        # Start cooldown, prolong current session.
                        cooldown_end_ts = current_frame.ts + cooldown_duration
                        self._mark_frame_with_session(_i, period_start_ts, True)

                    else:  # Previous frame is not "in progress".
                        if previous_frame.props["session_in_progress"]:  # Previous frame belongs to some session.
                            # Start new period with no session.
                            period_start_ts = current_frame.ts
                            self._mark_frame_with_session(_i, period_start_ts, False)
                        else:  # Previous frame does not belong to some session.
                            # Prolong current period with no session.
                            self._mark_frame_with_session(_i, period_start_ts, False)

            # Ensure that all combinations are covered.
            if not current_frame.props:
                raise RerollHistoryException("Frame %r not marked" % current_frame)

        # Write "period_end_ts" property to all frames.
        last_frame_index = len(self._timeline) - 1
        current_period_end_ts = self._end_ts
        self._timeline[last_frame_index].props["period_end_ts"] = current_period_end_ts
        for _i in range(last_frame_index - 1, -1, -1):
            previous_frame = self._timeline[_i + 1]
            current_frame = self._timeline[_i]
            if current_frame.props["period_start_ts"] != previous_frame.props["period_start_ts"]:
                current_period_end_ts = previous_frame.ts
            current_frame.props["period_end_ts"] = current_period_end_ts

        # Write "cooldown_in_progress" to frame props.
        for frame in self._timeline:
            if_cooldown_in_progress = frame.props["session_in_progress"] and \
                frame.state is not None and \
                frame.state not in self._in_progress_statuses
            frame.props["cooldown_in_progress"] = if_cooldown_in_progress

    def _read_max_ts(self, table_name: str):
        max_ts_query = f"""
            (
                SELECT toUnixTimestamp(MAX(ts)) AS max_ts
                FROM {table_name}
                WHERE (nanny_service_name = '{self._nanny_service_name}')
            )
        """
        row = next(self._clickhouse_client.select(max_ts_query))
        return row.max_ts

    def _read_timeline(self, start_ts: int, end_ts: int, wait_for_end_ts: bool):
        raise NotImplementedError

    def _wait_for_end_ts(self,
                         end_ts: int,
                         table_name: str,
                         max_tries: int = 18,
                         sleep_between_tries: int = 5):
        max_ts = self._read_max_ts(table_name)
        end_ts_present = max_ts >= end_ts
        num_tries = 0
        while not end_ts_present and num_tries < max_tries:
            self.logger.debug("Polling for timestamp %r, current MAX(ts) is %r..." % (end_ts, max_ts))
            time.sleep(sleep_between_tries)
            num_tries += 1
            max_ts = self._read_max_ts(table_name)
            end_ts_present = max_ts >= end_ts
        if not end_ts_present:
            raise RerollHistoryException("Have been waiting too long for timestamp %r, current MAX(ts) is %r" %
                                         (end_ts, max_ts))


class ReallocationHistory(BaseRerollHistory):
    logger = logging.getLogger("reallocation_history")

    _in_progress_statuses = ["IN_PROGRESS"]
    _ts_step = 60

    def _get_cooldown_duration_sec(self):
        return (self._config_interface.get_tentacles_group_reallocation_settings(self._nanny_service_name)
                ["cooldown_after_reallocation_min"]) * 60

    def _read_timeline(self, start_ts: int, end_ts: int, wait_for_end_ts: bool):
        table_name = NannyState.table_name()
        query = f"""
            (
                SELECT toUnixTimestamp(ts) AS ts,
                       toString(reallocation_state_status) AS reallocation_state_status,
                       toString(reallocation_taskgroup_id) AS reallocation_taskgroup_id,
                       toString(latest_snapshot_id) AS latest_snapshot_id
                FROM {table_name}
                WHERE (nanny_service_name = '{self._nanny_service_name}')
                    AND (ts >= {start_ts})
                    AND (ts <= {end_ts})
                ORDER BY ts ASC
            )
        """

        if wait_for_end_ts:
            self._wait_for_end_ts(end_ts=end_ts, table_name=table_name)

        data_provider = self._clickhouse_client.select(query)
        return ReallocationTimeline(data_provider=data_provider,
                                    start_ts=start_ts,
                                    end_ts=end_ts,
                                    ts_step=self._ts_step)


class RedeploymentHistory(BaseRerollHistory):
    logger = logging.getLogger("redeployment_history")

    _in_progress_statuses = ["PREPARING", "UPDATING"]
    _ts_step = 60

    def _get_cooldown_duration_sec(self):
        return (self._config_interface.get_tentacles_group_redeployment_settings(self._nanny_service_name)
                ["cooldown_after_redeployment_min"]) * 60

    def _read_timeline(self, start_ts: int, end_ts: int, wait_for_end_ts: bool):
        table_name = NannyState.table_name()
        query = f"""
            (
                SELECT toUnixTimestamp(ts) AS ts,
                       toString(current_state) AS current_state,
                       toString(latest_snapshot_taskgroup_id) AS latest_snapshot_taskgroup_id,
                       toString(latest_snapshot_id) AS latest_snapshot_id
                FROM {table_name}
                WHERE (nanny_service_name = '{self._nanny_service_name}')
                    AND (ts >= {start_ts})
                    AND (ts <= {end_ts})
                ORDER BY ts ASC
            )
        """

        if wait_for_end_ts:
            self._wait_for_end_ts(end_ts=end_ts, table_name=table_name)

        data_provider = self._clickhouse_client.select(query)
        return RedeploymentTimeline(data_provider=data_provider,
                                    start_ts=start_ts,
                                    end_ts=end_ts,
                                    ts_step=self._ts_step)
