import typing

from dataclasses import dataclass, field


@dataclass
class TimelineFrame:
    ts: int
    state: typing.Union[str, None]
    taskgroup_id: typing.Union[str, None]
    latest_snapshot_id: typing.Union[str, None]
    props: dict = field(default_factory=dict)


class BaseRerollTimeline:
    def __init__(self,
                 data_provider: typing.Iterator,
                 start_ts: int,
                 end_ts: int,
                 ts_step: int = 60):
        """
        Container class for TimelineFrame elements.
        Reads rows from data_provider, converts them to TimelineFrame
        objects, fills missing rows with empty TimelineFrames.
        """
        self._start_ts = start_ts
        self._end_ts = end_ts
        self._ts_step = ts_step
        self.timeline: typing.List[TimelineFrame] = self._load_data(data_provider)

    def __getitem__(self, i):
        return self.timeline[i]

    def __len__(self):
        return len(self.timeline)

    def get_frames(self, start_ts: int, end_ts: int):
        return [
            frame
            for frame in self.timeline
            if start_ts <= frame.ts <= end_ts
        ]

    def get_oldest_non_empty_frame_index(self):
        for i in range(0, len(self.timeline)):
            if self.timeline[i].state is not None:
                return i
        raise ValueError("All frames in timeline are empty")

    def _get_row_state(self, row):
        raise NotImplementedError

    def _get_row_taskgroup_id(self, row):
        raise NotImplementedError

    def _load_data(self, data_provider: typing.Iterator):
        timeline = []
        current_ts = self._start_ts

        for row in data_provider:

            while current_ts < row.ts:
                timeline.append(TimelineFrame(ts=current_ts, state=None, taskgroup_id=None, latest_snapshot_id=None))
                current_ts += self._ts_step

            timeline.append(TimelineFrame(ts=current_ts, state=self._get_row_state(row),
                                          taskgroup_id=self._get_row_taskgroup_id(row),
                                          latest_snapshot_id=row.latest_snapshot_id))
            current_ts += self._ts_step

        while current_ts <= self._end_ts:
            timeline.append(TimelineFrame(ts=current_ts, state=None, taskgroup_id=None, latest_snapshot_id=None))
            current_ts += self._ts_step

        return timeline


class ReallocationTimeline(BaseRerollTimeline):
    def _get_row_state(self, row):
        return row.reallocation_state_status

    def _get_row_taskgroup_id(self, row):
        return row.reallocation_taskgroup_id


class RedeploymentTimeline(BaseRerollTimeline):
    def _get_row_state(self, row):
        return row.current_state

    def _get_row_taskgroup_id(self, row):
        return row.latest_snapshot_taskgroup_id
