import enum

import requests

from infra.rtc_sla_tentacles.backend.lib.util import requests_retry_session
from infra.rtc_sla_tentacles.backend.lib.juggler_checks_manager import util as checks_util


class JugglerStatus(enum.Enum):
    OK = "OK"
    CRIT = "CRIT"
    WARN = "WARN"


class JugglerSender:

    PUSH_URL = 'http://127.0.0.1:31579/events'

    def __init__(self, logger, config_interface):
        self._logger = logger
        self._session = requests_retry_session()
        self._env_name = config_interface.get_env_name()

    def ok(self, *args, **kwargs):
        return self.send(JugglerStatus.OK, *args, **kwargs)

    def warn(self, *args, **kwargs):
        return self.send(JugglerStatus.WARN, *args, **kwargs)

    def crit(self, *args, **kwargs):
        return self.send(JugglerStatus.CRIT, *args, **kwargs)

    def send(self, status: JugglerStatus,
             *,
             service: str,
             host: str = None,
             instance: str = "",
             description: str = ""):
        # TODO(rocco66): extract sending to separate loop
        if self._env_name not in ["production", "testing"]:
            return
        event_dict = {
            "status": status.value,
            "service": checks_util.add_event_prefix(self._env_name, service),
            "instance": instance,
            "description": description,
            "tags": [f"tentacles_{self._env_name}"]
        }
        if host:
            # NOTE(rocco66): if host is not specified, then it will be equal container name
            event_dict["host"] = checks_util.add_event_prefix(self._env_name, host)
        response = self._session.post('http://127.0.0.1:31579/events', json={"events": [event_dict]})
        try:
            response.raise_for_status()
        except requests.exceptions.HTTPError:
            self._logger.warning("Can't send event to juggler subagent")
            return
        try:
            answer = response.json()
        except ValueError:
            self._logger.warning(f"can't parse subagent response: {response.text}")
            return
        if not answer.get("success", False):
            if 'message' in answer:
                self._logger.warning('send_to_juggler failed: %s', answer['message'])
            else:
                self._logger.warning('send_to_juggler failed: unknown error')
