import time
import typing

from dataclasses import dataclass


Histogram = typing.List[typing.List[int]]


@dataclass
class TimestampFileReadResult:
    ts: typing.Optional[int] = None
    errmsg: typing.Optional[str] = None

    def __str__(self):
        return str(self.ts) if self.ts else self.errmsg


class TentacleState:
    """
    Reads:
    * timestamp from "timestamp resource" file
    * timestamp from "timestamp resource prepare finish time" file
    * timestamp from "self shutdown time" file

    Returns:
    * Golovan "resource delivery duration" histogram - time between
      now and prepare hook execution time
    * Golovan "configuration switch duration" histogram - time
      between now and time in timestamp resource file
    * Golovan "activation duration" histogram - time between now
      and self shutdown time on activation phase
    * Juggler 'timestamp too old' check

    All these values are static during binary lifetime.
    """
    def __init__(self,
                 binary_start_ts: int,
                 timestamp_files_settings: dict,
                 juggler_settings: dict):
        self._binary_start_ts = binary_start_ts

        self._resource_timestamp = self._read_ts_file(timestamp_files_settings["resource_file_path"])
        self._prepare_finish_time_timestamp = self._read_ts_file(
            timestamp_files_settings["prepare_finish_time_file_path"])
        self._self_shutdown_time_timestamp = self._read_ts_file(
            timestamp_files_settings["self_shutdown_time_file_path"])

        self._juggler_settings = juggler_settings

        self.resource_delivery_duration_histogram = [
            bucket
            for bucket in self._get_buckets(
                borders=self._get_borders(start=0, stop=1200, step=30, rightmost_border=60*60*24*31),
                value=self._get_resource_delivery_duration()
            )
        ]
        self.configuration_switch_duration_histogram = [
            bucket
            for bucket in self._get_buckets(
                borders=self._get_borders(start=0, stop=1200, step=30, rightmost_border=60*60*24*31),
                value=self._get_configuration_switch_duration()
            )
        ]
        self.activation_duration_histogram = [
            bucket
            for bucket in self._get_buckets(
                borders=self._get_activation_duration_buckets_borders(),
                value=self._get_activation_duration()
            )
        ]

    def get_timestamps(self) -> dict:
        return {
            "resource_timestamp": str(self._resource_timestamp),
            "prepare_finish_time_timestamp": str(self._prepare_finish_time_timestamp),
            "self_shutdown_time_timestamp": str(self._self_shutdown_time_timestamp),
            "resource_delivery_duration": self._get_resource_delivery_duration(),
            "configuration_switch_duration": self._get_configuration_switch_duration(),
            "activation_duration": self._get_activation_duration()
        }

    def get_juggler_timestamp_age_monitoring(self) -> (str, int):
        if not self._resource_timestamp.ts:
            return self._resource_timestamp.errmsg, self._juggler_settings["crit_code"]
        if time.time() - self._resource_timestamp.ts >= self._juggler_settings["crit_age"]:
            return "Timestamp too old", self._juggler_settings["crit_code"]
        return "Ok", self._juggler_settings["ok_code"]

    def get_golovan_data(self) -> typing.List[typing.List[typing.Union[str, Histogram]]]:
        result = list()
        result.append(["v1_resource_delivery_duration_ahhh", self.resource_delivery_duration_histogram])
        result.append(["v1_configuration_switch_duration_ahhh", self.configuration_switch_duration_histogram])
        result.append(["v1_activation_duration_ahhh", self.activation_duration_histogram])
        return result

    def _get_resource_delivery_duration(self) -> typing.Optional[int]:
        if not self._prepare_finish_time_timestamp.ts or not self._resource_timestamp.ts:
            return None
        return self._prepare_finish_time_timestamp.ts - self._resource_timestamp.ts

    def _get_configuration_switch_duration(self) -> typing.Optional[int]:
        if not self._resource_timestamp.ts:
            return None
        return self._binary_start_ts - self._resource_timestamp.ts

    def _get_activation_duration(self) -> typing.Optional[int]:
        if not self._self_shutdown_time_timestamp.ts:
            return None
        return self._binary_start_ts - self._self_shutdown_time_timestamp.ts

    @staticmethod
    def _get_buckets(borders: typing.Iterator,
                     value: typing.Optional[int]) -> typing.Iterator:
        yielded = False
        left_border = next(borders)
        for right_border in borders:
            if value is not None and not yielded and left_border <= value < right_border:
                yield [left_border, 1]
                yielded = True
            else:
                yield [left_border, 0]
            left_border = right_border

    @staticmethod
    def _read_ts_file(file_path: str) -> TimestampFileReadResult:
        try:
            with open(file_path, "r") as f:
                return TimestampFileReadResult(ts=int(f.read()))
        except Exception as _exc:
            return TimestampFileReadResult(errmsg="ERROR: " + str(_exc))

    @classmethod
    def _get_borders(cls,
                     start: int,
                     stop: int,
                     step: int,
                     rightmost_border: int = 0) -> typing.Iterator:
        for i in range(start, stop, step):
            yield i
        yield stop
        if rightmost_border:
            yield rightmost_border

    @classmethod
    def _get_activation_duration_buckets_borders(cls) -> typing.Iterator:
        for i in cls._get_borders(start=0, stop=30, step=1):
            yield i
        for i in cls._get_borders(start=35, stop=90, step=5, rightmost_border=60*60*24*31):
            yield i
