# encoding: utf-8
import functools
import io
import json
import os
import logging
import time

import flask
import requests
import sentry_sdk
import sentry_sdk.integrations.logging
import sentry_sdk.integrations.flask
import errorboosterclient.sentry
from logbroker.unified_agent.client import python as unified_agent

import typing
import yaml
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry


UNIFIED_AGENT_URI = "localhost:16302"


class PathError(KeyError):
    def __init__(self, part, path):
        self.part = part
        self.path = '/'.join(path)

    def __str__(self):
        return f'{self.part!r} not found in {self.path!r}'


def get_value_in_dict_by_path_or_raise(tree, key_name, *path_holding_key_name):
    current_path = tree
    human_readable_current_path = ["tree_root"]

    for part in path_holding_key_name:
        if part in current_path:
            human_readable_current_path.append(part)
            current_path = current_path[part]
        else:
            raise PathError(part, human_readable_current_path)
    if key_name in current_path:
        return current_path[key_name]
    else:
        raise PathError(key_name, human_readable_current_path)


def modify_value_in_dict_by_path_or_raise(tree, key_name, value_to_set, *path_holding_key_name):
    current_path = tree
    human_readable_current_path = ["tree_root"]

    for part in path_holding_key_name:
        if part in current_path:
            current_path = current_path[part]
            human_readable_current_path.append(part)
        else:
            raise PathError(part, human_readable_current_path)

    if key_name in current_path:
        current_path[key_name] = value_to_set
    else:
        raise PathError(key_name, human_readable_current_path)


def read_yaml_file(stream):
    if not isinstance(stream, io.IOBase):
        cleanpath = os.path.abspath(stream)
        stream = open(cleanpath, "r")
    contents = yaml.safe_load(stream)
    return contents


def retry(retries: int = 3, backoff: float = 2, catched_exceptions=(Exception,), logger=None):

    def decorator(func):

        @functools.wraps(func)
        def inner(*args, **kwargs):
            delay = 0.5
            _retries = retries
            while _retries > 1:
                try:
                    return func(*args, **kwargs)
                except catched_exceptions as exc:
                    msg = "%s while try %s, retrying in %d seconds" % (str(exc), func.__name__, delay)
                    if logger:
                        logger.warning(msg)
                    time.sleep(delay)
                    _retries -= 1
                    delay *= backoff
            return func(*args, **kwargs)

        return inner

    return decorator


def requests_retry_session(retries: int = 3,
                           backoff_factor: float = 0.5,
                           status_forcelist: typing.Iterable[int] = (requests.codes.TOO_MANY_REQUESTS,  # 429
                                                                     requests.codes.INTERNAL_SERVER_ERROR,  # 500
                                                                     requests.codes.BAD_GATEWAY,  # 502
                                                                     requests.codes.GATEWAY_TIMEOUT),  # 504
                           session: typing.Optional[requests.Session] = None):
    session = session or requests.Session()
    retry = Retry(total=retries,
                  redirect=1,
                  backoff_factor=backoff_factor,
                  status_forcelist=status_forcelist)
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


class SentrySender:

    def __init__(self, manual_init=False):
        if not manual_init:
            self.init()
        self.environment = "unknown"

    def send_to_errorbuster(self, event: dict):
        event["env"] = self.environment
        self._session.send(json.dumps(event), time.time(), event)

    def init(self):
        client = unified_agent.Client(UNIFIED_AGENT_URI, log_level=logging.ERROR)
        self._session = client.create_session()
        sentry_sdk.init(
            transport=errorboosterclient.sentry.ErrorBoosterTransport(
                project='tentacles',
                sender=self.send_to_errorbuster,
            ),
            integrations=[
                sentry_sdk.integrations.logging.LoggingIntegration(level=logging.ERROR),
                sentry_sdk.integrations.flask.FlaskIntegration(),
            ],
        )


def get_flask_server(name):
    app = flask.Flask(name)

    @app.route('/ping', methods=['GET'])
    def ping():
        return "OK"

    return app
