import typing

from infra.nanny.yp_lite_api.py_stubs.pod_sets_api_stub import YpLiteUIPodSetsServiceStub
from infra.nanny.yp_lite_api.proto import pod_sets_api_pb2
from infra.rtc_sla_tentacles.backend.lib.funccall_stats_server import server as stat_server


class AllocationRequestProviderException(Exception):
    pass


class AllocationRequestProvider:
    """
        Reads 'pod_sets_api_pb2.AllocationRequest' of a given
        Nanny services' pod from Nanny YP Lite API.
        If no 'pod_filter' given, reads 'AllocationRequest'
        of some random pod.
    """
    def __init__(self, nanny_service_name: str, yp_cluster: str,
                 yp_lite_ui_podsets_service_stub: YpLiteUIPodSetsServiceStub, pod_id: str = None):
        self._nanny_service_name = nanny_service_name
        self._yp_cluster = yp_cluster
        self._yp_lite_ui_podsets_service_stub = yp_lite_ui_podsets_service_stub

        if pod_id is None:
            pod_id = self._get_some_pod_id()
        self._pod_id = pod_id

        self._allocation_request = self._load_allocation_request()

    def get_allocation_request(self) -> pod_sets_api_pb2.AllocationRequest:
        return self._allocation_request

    def get_pod_specific_allocation_request(self, node_id: str, strong: bool = False) \
            -> pod_sets_api_pb2.PodSpecificAllocationRequest:
        # Create empty 'PodSpecificSettings' message.
        pod_specific_settings = pod_sets_api_pb2.PodSpecificSettings()
        # pod_specific_settings.pod_id = self._pod_id
        # Set scheduling hint.
        scheduling_hint = pod_specific_settings.scheduling.hints.add()
        scheduling_hint.node_id = node_id
        scheduling_hint.strong = strong
        # Create 'PodSpecificAllocationRequest'.
        pod_specific_allocation_request = pod_sets_api_pb2.PodSpecificAllocationRequest()
        pod_specific_allocation_request.allocation.CopyFrom(self._allocation_request)
        pod_specific_allocation_request.allocation.replicas = 1
        pod_specific_allocation_request.pod_specific_settings.CopyFrom(pod_specific_settings)
        return pod_specific_allocation_request

    def flip_root_fs_quota_megabytes(self):
        self._allocation_request.root_fs_quota_megabytes ^= 1

    def set_volumes_storage_class(self, storage_class: str):
        self._allocation_request.root_volume_storage_class = storage_class
        for volume_request in self._allocation_request.persistent_volumes:
            volume_request.storage_class = storage_class

    def set_volumes_bandwidth_limit(self, root_limit_megabytes_per_sec: int = 10,
                                    persistent_volumes_limit_megabytes_per_sec: int = 10):
        self._allocation_request.root_bandwidth_limit_megabytes_per_sec = root_limit_megabytes_per_sec
        for volume_request in self._allocation_request.persistent_volumes:
            volume_request.bandwidth_limit_megabytes_per_sec = persistent_volumes_limit_megabytes_per_sec

    def set_network_bandwidth_limit(self, limit_megabytes_per_sec: int = 10):
        self._allocation_request.network_bandwidth_limit_megabytes_per_sec = limit_megabytes_per_sec

    def _get_some_pod_id(self) -> str:
        list_pods_request = pod_sets_api_pb2.ListPodsRequest()
        list_pods_request.service_id = self._nanny_service_name
        list_pods_request.cluster = self._yp_cluster
        list_pods_request.field_mask.paths.extend(["meta.id"])
        list_pods_request.offset = 0
        list_pods_request.limit = 1
        with stat_server.yp_lite_timing():
            list_pods_response = self._yp_lite_ui_podsets_service_stub.list_pods(list_pods_request)
        if list_pods_response.total:
            return list_pods_response.pods[0].meta.id

    def _load_allocation_request(self) -> typing.Optional[pod_sets_api_pb2.AllocationRequest]:
        list_pods_groups_request = pod_sets_api_pb2.ListPodsGroupsRequest()
        list_pods_groups_request.service_id = self._nanny_service_name
        list_pods_groups_request.cluster = self._yp_cluster
        list_pods_groups_request.pod_filter = f"[/meta/id]=\"{self._pod_id}\""
        with stat_server.yp_lite_timing():
            list_pods_group_response = self._yp_lite_ui_podsets_service_stub.list_pods_groups(list_pods_groups_request)
        try:
            pods_group = list_pods_group_response.pods_groups[0]
        except KeyError:
            raise AllocationRequestProviderException("Empty 'ListPodsGroupsResponse' reply on pod_filter '%s'" %
                                                     list_pods_groups_request.pod_filter)
        return pods_group.allocation_request
