import collections
import enum
import logging
import time
import typing

from collections import defaultdict
from dataclasses import dataclass
from random import shuffle

from infra.nanny.nanny_services_rest.nanny_services_rest.client import ServiceRepoClient
from infra.nanny.yp_lite_api.proto import pod_sets_api_pb2
from infra.nanny.yp_lite_api.py_stubs.pod_sets_api_stub import YpLiteUIPodSetsServiceStub
from nanny_rpc_client import RetryingRpcClient
from yp.client import YpClient, BatchingOptions

from infra.rtc_sla_tentacles.backend.lib.clickhouse.client import ClickhouseClient
from infra.rtc_sla_tentacles.backend.lib.clickhouse.database import YpLitePods
from infra.rtc_sla_tentacles.backend.lib.config.interface import ConfigInterface
from infra.rtc_sla_tentacles.backend.lib.yp_lite.allocation import AllocationRequestProvider
from infra.rtc_sla_tentacles.backend.lib.funccall_stats_server import server as stat_server


class PodDiskReq(str, enum.Enum):
    PodDiskReqHdd = "hdd"
    PodDiskReqSsd = "ssd"


@dataclass
class NodeInfo:
    node_id: str
    hfsm_state: str


@dataclass
class PodInfo:
    fqdn: str
    node_id: str
    scheduling_hint_node_id: typing.Optional[str]
    yp_pod_disk_req: typing.Optional[PodDiskReq]


@dataclass(eq=True)
class YpLitePodsManagerPodData:
    yp_node_id: typing.Optional[str] = None
    yp_node_has_hdd: typing.Optional[bool] = None
    yp_node_has_ssd: typing.Optional[bool] = None
    yp_node_hfsm_state: typing.Optional[str] = None
    yp_node_matches_node_filter: bool = False
    yp_pod_id: typing.Optional[str] = None
    yp_pod_fqdn: typing.Optional[str] = None
    yp_pod_disk_req: typing.Optional[PodDiskReq] = None
    nanny_instance_pod_id: typing.Optional[str] = None
    scheduling_hint_node_id: typing.Optional[str] = None


class YpLitePodsManagerException(Exception):
    pass


class YpLitePodsManager:
    def __init__(self, nanny_service_name: str, yp_cluster: str, logger: logging.Logger, *,
                 restrict_nodes_to_specified_in_podset: bool = False,
                 config_interface: ConfigInterface = None,
                 init_yp_client=False,
                 init_nanny_podsets_service_stub=False,
                 init_nanny_service_repo_client=False,
                 init_clickhouse_client=False,
                 ):
        self._logger = logger
        self._yp_cluster = yp_cluster
        self._restrict_nodes_to_specified_in_podset = restrict_nodes_to_specified_in_podset
        self._nanny_service_name = nanny_service_name
        self._config_interface = config_interface or ConfigInterface()

        self._config = self._config_interface.get_yp_lite_pods_manager_config()

        self._yp_client = None
        self._nanny_podsets_service_stub = None
        self._nanny_service_repo_client = None
        self._clickhouse_client = None

        self._init_yp_client = init_yp_client
        self._init_nanny_podsets_service_stub = init_nanny_podsets_service_stub
        self._init_nanny_service_repo_client = init_nanny_service_repo_client
        self._init_clickhouse_client = init_clickhouse_client

        self._pods_data = []
        self._stats = {}

    def __enter__(self):
        if self._init_yp_client:
            config_kwargs = self._config_interface.get_yp_client_config(self._yp_cluster)
            self._yp_client = YpClient(**config_kwargs)

        if self._init_nanny_podsets_service_stub:
            config_kwargs = self._config_interface.get_nanny_rpc_client_config(rpc_url_name="yp_lite_pod_sets_url")
            self._nanny_podsets_service_stub = YpLiteUIPodSetsServiceStub(client=RetryingRpcClient(**config_kwargs))

        if self._init_nanny_service_repo_client:
            config_kwargs = self._config_interface.get_nanny_rest_client_config()
            self._nanny_service_repo_client = ServiceRepoClient(**config_kwargs)

        if self._init_clickhouse_client:
            self._clickhouse_client = ClickhouseClient(self._config_interface)

        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self._yp_client:
            self._yp_client.close()
        if self._nanny_podsets_service_stub:
            self._nanny_podsets_service_stub.client.close()
        if self._nanny_service_repo_client:
            self._nanny_service_repo_client._session.close()  # noqa
        if self._clickhouse_client:
            db_connect = self._clickhouse_client._db
            if db_connect:
                db_connect.request_session.close()  # noqa

    def get_podset_node_filter(self) -> str:
        return self._get_yp_podset_node_filter()

    def get_stats(self) -> typing.Dict[str, int]:
        """
            Returns a dictionary with data freshness timestamp and
            numbers of nodes, pods and Nanny instances.
        """
        if not self._stats:
            raise YpLitePodsManagerException("No pods data loaded yet!")
        return self._stats

    def insert_allocated_pods_to_nanny_runtime_attributes(self, active_runtime_attrs: dict):
        """
            Inserts allocated pods to given Nanny service
            configuration dict.
        """
        new_nanny_instances_list = [
            {
                "cluster": self._yp_cluster,
                "pod_id": pod_id
            }
            for pod_id in self._get_actual_allocated_pods()
        ]
        if not new_nanny_instances_list:
            raise YpLitePodsManagerException("No allocated pods found")
        active_runtime_attrs["content"]["instances"]["yp_pod_ids"]["pods"] = new_nanny_instances_list
        return active_runtime_attrs

    def fetch_yp_nodes_info(self) -> typing.Dict[str, NodeInfo]:
        return {
            node_id: NodeInfo(node_id, hfsm_state)
            for node_id, hfsm_state
            in self.fetch_yp_nodes(selectors=["/meta/id", "/status/hfsm/state"])
        }

    def load_data_from_api(self):
        """
            Loads nodes and pods data from APIs.
        """
        yp_nodes = self.fetch_yp_nodes_info()
        yp_nodes_resources = self._fetch_yp_nodes_resources(yp_nodes)
        pods = self._fetch_pods()
        nanny_instances = self._fetch_nanny_instances()
        self._pods_data = self._merge(yp_nodes, yp_nodes_resources, pods, nanny_instances)
        self._generate_pods_stats(freshness_ts=int(time.time()))

    def load_pods_from_clickhouse(self, ts_to_wait_for: int = 0):
        """
            Loads most recent pods data from ClickHouse.
            If `ts_to_wait_for` given, wait for MAX(ts) in table to
            become even or greater than `ts_to_wait_for`.
        """
        table_name = YpLitePods.table_name()
        query = f"""
            (
                SELECT toUnixTimestamp(ts) AS ts,
                       toString(yp_node_id) AS yp_node_id,
                       toString(yp_pod_id) AS yp_pod_id,
                       toString(nanny_instance_pod_id) AS nanny_instance_pod_id,
                       toString(scheduling_hint_node_id) AS scheduling_hint_node_id,
                       yp_pod_disk_req,
                       yp_node_has_hdd,
                       yp_node_has_ssd,
                       yp_node_matches_node_filter,
                       yp_node_hfsm_state
                FROM {table_name}
                WHERE (nanny_service_name = '{self._nanny_service_name}')
                    AND (ts = (SELECT MAX(ts)
                               FROM {table_name}
                               WHERE (nanny_service_name = '{self._nanny_service_name}')
                              )
                        )
            )
        """
        if ts_to_wait_for:
            self._wait_for_ts(ts_to_wait_for)

        query_result = self._clickhouse_client.select(query)
        if not query_result:
            raise YpLitePodsManagerException("No rows received from ClickHouse")

        pods_data = []
        ts = None
        for row in query_result:
            if not ts:
                ts = row.ts
            pods_data.append(YpLitePodsManagerPodData(
                yp_node_id=row.yp_node_id,
                yp_node_matches_node_filter=row.yp_node_matches_node_filter,
                yp_pod_id=row.yp_pod_id,
                nanny_instance_pod_id=row.nanny_instance_pod_id,
                scheduling_hint_node_id=row.scheduling_hint_node_id,
                yp_pod_disk_req=row.yp_pod_disk_req and PodDiskReq(row.yp_pod_disk_req.name),
                yp_node_has_hdd=row.yp_node_has_hdd,
                yp_node_has_ssd=row.yp_node_has_ssd,
                yp_node_hfsm_state=row.yp_node_hfsm_state,
            ))
        self._logger.debug("Loaded '%d' records from ClickHouse, ts='%d'" % (len(pods_data), ts))
        self._pods_data = pods_data
        self._generate_pods_stats(freshness_ts=ts)

    def remove_pod(self, pod_id):
        self._logger.debug("Removing pod '%s'" % pod_id)
        remove_pod_request = pod_sets_api_pb2.RemovePodRequest()
        remove_pod_request.pod_id = pod_id
        remove_pod_request.cluster = self._yp_cluster
        with stat_server.yp_lite_timing():
            self._nanny_podsets_service_stub.remove_pod(remove_pod_request)

    def update_pods(self):
        # Delete pods that won't be ever allocated, and quit.
        pods_to_delete = self._get_pods_scheduled_on_non_existing_nodes()
        pods_to_delete += self._get_pods_with_incorrect_scheduling_hints()
        self._logger.info(f"Pods for removing {', '.join(pods_to_delete)}")
        if pods_to_delete:
            shuffle(pods_to_delete)
            del pods_to_delete[self._config["pods_without_nodes_removal_batch_size"]:]
            self._logger.info(("Removing '%d' pods that are not allocated, have non-existing node in scheduling hint, "
                               "and not in Nanny instances") % len(pods_to_delete))
            for pod_id in pods_to_delete:
                try:
                    self.remove_pod(pod_id)
                except Exception:
                    self._logger.exception("Exception while removing pod '%s'", pod_id)
                    continue
            self._logger.info("Successfully removed '%d' pods" % len(pods_to_delete))
            return

        # Check that there are nodes without pods and not in pods' scheduling hints, if no such nodes - quit.
        nodes_to_allocate_pods_on = list(self._get_nodes_without_pods_and_not_in_scheduling_hints())
        if not nodes_to_allocate_pods_on:
            self._logger.info("No pods to remove, no nodes to allocate pods on, exiting")
            return

        # Allocate new pods on nodes without pods.
        shuffle(nodes_to_allocate_pods_on)
        del nodes_to_allocate_pods_on[self._config["new_pods_allocation_batch_size"]:]
        self._logger.info("Allocating new pods on '%d' nodes" % len(nodes_to_allocate_pods_on))
        nodes_with_storage_class = self._fetch_nodes_disks_storage_class(nodes_to_allocate_pods_on)
        allocation_request_provider = AllocationRequestProvider(self._nanny_service_name, self._yp_cluster,
                                                                self._nanny_podsets_service_stub)
        for node, storage_classes in nodes_with_storage_class.items():
            try:
                self._allocate_pod_on_node(allocation_request_provider, node, storage_classes)
            except Exception as _exc:
                self._logger.warning("Exception while allocation pod with scheduling hint '%s': '%s'" %
                                     (node, _exc))
                continue
        self._logger.info("Successfully allocated '%d' pods" % len(nodes_with_storage_class))

    def update_nanny_instances(self):
        """
            Updates instances list in Nanny service configuration
            with allocated pods.
        """
        with stat_server.nanny_timing():
            active_runtime_attrs = self._nanny_service_repo_client.get_active_runtime_attrs(self._nanny_service_name)
        active_runtime_attrs = self.insert_allocated_pods_to_nanny_runtime_attributes(active_runtime_attrs)
        request = {
            "snapshot_id": active_runtime_attrs["_id"],
            "content": active_runtime_attrs["content"],
            "comment": "Update instances list",
            "meta_info": {
                "scheduling_config": {
                    "scheduling_priority": "CRITICAL",
                },
            }
        }
        self._logger.info("Updating Nanny service instances list with '%d' allocated pods" %
                          len(active_runtime_attrs["content"]["instances"]["yp_pod_ids"]["pods"]))
        with stat_server.nanny_timing():
            self._nanny_service_repo_client.put_runtime_attrs(s_id=self._nanny_service_name, content=request)
        self._logger.info("Updated Nanny service instances list with '%d' allocated pods" %
                          len(active_runtime_attrs["content"]["instances"]["yp_pod_ids"]["pods"]))

    def set_node_filter_rack_string_comparison(self, node_percent: int):
        self._logger.info(("Setting node filter for podset '%s' to rack name string comparison, no less than '%d' "
                           "percent of nodes") % (self.get_podset_name(), node_percent))
        node_segment_id = self._get_yp_node_segment_id()
        node_segment_node_filter = self._get_yp_node_segment_node_filter(node_segment_id)

        racks = list(self.select_objects(
            "node",
            filter=node_segment_node_filter,
            selectors=["/labels/topology/rack"],
        ))
        num_nodes_in_rack = defaultdict(int)
        for rack in racks:
            try:
                num_nodes_in_rack[rack] += 1
            # Ignore nodes with empty '/labels/topology/rack', see https://paste.yandex-team.ru/1140033
            except TypeError:
                pass
        total_nodes = sum(num_nodes_in_rack.values())

        num_nodes_behind_filter = 0
        rack_to_put_into_node_filter = ""
        for rack in sorted(num_nodes_in_rack.keys()):
            num_nodes_behind_filter += num_nodes_in_rack[rack]
            if num_nodes_behind_filter >= (total_nodes * node_percent / 100):
                rack_to_put_into_node_filter = rack
                break

        if not rack_to_put_into_node_filter:
            raise YpLitePodsManagerException("Error looking for rack covering '%d' nodes" % node_percent)

        node_filter = f'string([/labels/topology/rack]) <= "{rack_to_put_into_node_filter}"'
        self._logger.info("Node filter '%s' covers '%d' nodes" % (node_filter, num_nodes_behind_filter))
        self._set_yp_podset_node_filter(node_filter)

    def set_empty_node_filter(self):
        self._logger.info("Setting empty node filter for podset '%s'." % self.get_podset_name())
        node_filter = ""
        self._set_yp_podset_node_filter(node_filter)

    def write_pods_to_clickhouse(self, ts: int):
        """
            Writes pods data to ClickHouse with given timestamp.
        """
        output = [YpLitePods(
            ts=ts,
            nanny_service_name=self._nanny_service_name,
            yp_cluster=self._yp_cluster,
            yp_node_id=pod_data.yp_node_id,
            yp_node_matches_node_filter=pod_data.yp_node_matches_node_filter,
            yp_node_has_hdd=pod_data.yp_node_has_hdd,
            yp_node_has_ssd=pod_data.yp_node_has_ssd,
            yp_node_hfsm_state=pod_data.yp_node_hfsm_state,

            yp_pod_id=pod_data.yp_pod_id,
            yp_pod_fqdn=pod_data.yp_pod_fqdn,
            yp_pod_disk_req=pod_data.yp_pod_disk_req,
            nanny_instance_pod_id=pod_data.nanny_instance_pod_id,
            scheduling_hint_node_id=pod_data.scheduling_hint_node_id,
        ) for pod_data in self._pods_data]
        self._logger.debug("Dumping '%d' records to ClickHouse, ts='%d'" % (len(output), ts))
        self._clickhouse_client.insert(output)

    def _allocate_pod_on_node(self, allocation_request_provider: AllocationRequestProvider,
                              node: str, storage_classes: typing.Set[str]):
        # PLN-408 Prefer allocating on HDD.
        if "hdd" not in storage_classes and "ssd" in storage_classes:
            storage_class_to_allocate_on = "ssd"
        else:
            storage_class_to_allocate_on = "hdd"
        self._logger.info("Allocating pod on node '%s', storage class '%s'" % (node, storage_class_to_allocate_on))
        allocation_request_provider.set_volumes_storage_class(storage_class_to_allocate_on)
        allocation_request_provider.set_volumes_bandwidth_limit()
        allocation_request_provider.set_network_bandwidth_limit()
        pod_specific_allocation_request = allocation_request_provider.get_pod_specific_allocation_request(node, True)
        create_specific_pod_request = pod_sets_api_pb2.CreateSpecificPodRequest()
        create_specific_pod_request.service_id = self._nanny_service_name
        create_specific_pod_request.allocation_request.CopyFrom(pod_specific_allocation_request)
        create_specific_pod_request.cluster = self._yp_cluster
        create_specific_pod_response = pod_sets_api_pb2.CreateSpecificPodResponse()
        rpc_client = self._nanny_podsets_service_stub.client
        with stat_server.yp_lite_timing():
            rpc_client.call_remote_method("CreateSpecificPod", create_specific_pod_request, create_specific_pod_response)
        self._logger.info("Allocated pod '%s' on node '%s', storage class '%s'" %
                          (create_specific_pod_response.pod_id, node, storage_class_to_allocate_on))

    def _generate_pods_stats(self, freshness_ts: int):
        self._stats["data_freshness"] = freshness_ts

        self._stats["pods"] = {}
        pods = self._stats["pods"]
        pods["pods_seen"] = self._get_num_of_rows_matched(self._row_matcher_pods_seen)
        pods["pods_allocated"] = self._get_num_of_rows_matched(self._row_matcher_pods_actual_allocated)
        pods["pods_not_allocated"] = self._get_num_of_rows_matched(self._row_matcher_pods_not_allocated)
        pods["pods_in_deploy_system"] = self._get_num_of_rows_matched(self._row_matcher_pods_in_deploy_system)
        pods["pods_not_in_deploy_system"] = self._get_num_of_rows_matched(self._row_matcher_pods_not_in_deploy_system)
        pods["pods_allocated_and_in_deploy_system"] = self._get_num_of_rows_matched(
            self._row_matcher_pods_allocated_and_in_deploy_system)
        pods["pods_not_allocated_and_not_in_deploy_system"] = self._get_num_of_rows_matched(
            self._row_matcher_pods_not_allocated_and_not_in_deploy_system)
        pods["pods_scheduled_on_non_existing_nodes"] = len(self._get_pods_scheduled_on_non_existing_nodes())
        pods["incorrect_scheduling_hints"] = len(self._get_pods_with_incorrect_scheduling_hints())

        self._stats["nodes"] = {}
        nodes = self._stats["nodes"]
        nodes["nodes_seen"] = self._get_num_of_rows_matched(self._row_matcher_nodes_seen)
        nodes["nodes_up_seen"] = self._get_num_of_rows_matched(self._row_matcher_nodes_up_seen)
        nodes["nodes_without_pods"] = self._get_num_of_rows_matched(self._row_matcher_nodes_without_pods)
        nodes["nodes_without_pods_and_not_in_scheduling_hints"] = \
            len(self._get_nodes_without_pods_and_not_in_scheduling_hints())

        self._stats["nanny_instances"] = {}
        nanny_instances = self._stats["nanny_instances"]
        nanny_instances["nanny_instances_seen"] = self._get_num_of_rows_matched(self._row_matcher_nanny_instances_seen)
        nanny_instances["nanny_instances_without_pods"] = self._get_num_of_rows_matched(
            self._row_matcher_nanny_instances_without_pods)

    def _get_actual_allocated_pods(self) -> typing.Iterable[str]:
        return (
            row.yp_pod_id
            for row in self._pods_data
            if self._row_matcher_pods_actual_allocated(row)
        )

    def _get_nodes_without_pods(self) -> typing.List[str]:
        return [
            row.yp_node_id
            for row in self._pods_data
            if self._row_matcher_nodes_without_pods(row)
        ]

    def _get_num_of_rows_matched(self, row_matcher: typing.Callable) -> int:
        counter = 0
        for row in self._pods_data:
            if row_matcher(row):
                counter += 1
        return counter

    def _get_nodes_matched(self, row_matcher: typing.Callable) -> typing.Set[str]:
        return {
            row.yp_node_id
            for row in self._pods_data
            if row_matcher(row)
        }

    def _get_pods_matched(self, row_matcher: typing.Callable) -> typing.List[str]:
        return [
            row.yp_pod_id
            for row in self._pods_data
            if row_matcher(row)
        ]

    def _get_nodes_without_pods_and_not_in_scheduling_hints(self) -> typing.Set[str]:
        nodes_without_pods = self._get_nodes_matched(self._row_matcher_nodes_without_pods)
        # Do not check through all pods, check only through not allocated pods.
        scheduling_hints_of_not_allocated_pods = {
            row.scheduling_hint_node_id
            for row in self._pods_data
            if self._row_matcher_pods_not_allocated(row)
        }
        return {
            node
            for node in nodes_without_pods
            if node not in scheduling_hints_of_not_allocated_pods
        }

    def _get_pods_with_incorrect_scheduling_hints(self) -> typing.List[str]:
        # NOTE(rocco66): see https://st.yandex-team.ru/TENTACLES-278
        nodes = {
            p.yp_node_id: p for p in self._pods_data if p.yp_node_id
        }
        result = []
        for pod in self._pods_data:
            if pod.yp_node_id:
                continue
            hint_node = nodes.get(pod.scheduling_hint_node_id)
            if not hint_node:
                continue
            incorrect_hint = (
                pod.yp_pod_disk_req == PodDiskReq.PodDiskReqHdd and not hint_node.yp_node_has_hdd or
                pod.yp_pod_disk_req == PodDiskReq.PodDiskReqSsd and not hint_node.yp_node_has_ssd
            )
            if incorrect_hint:
                result.append(pod.yp_pod_id)
                self._logger.info("Incorrect pod spec %s (node %s)", pod, hint_node)
        return result

    def _get_pods_scheduled_on_non_existing_nodes(self) -> typing.List[str]:
        # Select pods that are also not in deploy system - this is needed because this method
        # is used to get pods that will be deleted. Pod must be removed from deploy system
        # before it can be deleted.

        not_allocated_pods = [
            row
            for row in self._pods_data
            if self._row_matcher_pods_not_allocated_and_not_in_deploy_system(row)
        ]
        pods_allocated_on_non_matched_node = [
            row
            for row in self._pods_data
            if self._row_matcher_pods_allocated_on_non_matched_node(row)
        ]
        # Do not check through all nodes, check only through nodes without pods.
        nodes_without_pods = self._get_nodes_matched(self._row_matcher_nodes_without_pods)
        return [
            pod.yp_pod_id
            for pod in not_allocated_pods + pods_allocated_on_non_matched_node
            if pod.scheduling_hint_node_id not in nodes_without_pods
        ]

    def get_podset_name(self):
        return self._nanny_service_name.replace("_", "-")

    def _get_yp_node_segment_id(self):
        with stat_server.yp_timing():
            return self._yp_client.get_object(
                object_type="pod_set",
                object_identity=self.get_podset_name(),
                selectors=["/spec/node_segment_id"]
            )[0]  # default

    def _get_yp_podset_node_filter(self):
        with stat_server.yp_timing():
            return self._yp_client.get_object(
                object_type="pod_set",
                object_identity=self.get_podset_name(),
                selectors=["/spec/node_filter"]
            )[0]  # [/meta/id] IN("foo.y.n", "bar.y.n")

    def _set_yp_podset_node_filter(self, node_filter_value: str):
        with stat_server.yp_timing():
            self._yp_client.update_object(
                object_type="pod_set",
                object_identity=self.get_podset_name(),
                set_updates=[dict(path="/spec/node_filter", value=node_filter_value,)],
            )

    def _get_yp_node_segment_node_filter(self, node_segment_id: str) -> str:
        with stat_server.yp_timing():
            return self._yp_client.get_object(
                object_type="node_segment",
                object_identity=node_segment_id,
                selectors=["/spec/node_filter"]
            )[0]  # [/labels/segment]="default"

    def _get_yp_node_filter(self) -> str:
        node_segment_id = self._get_yp_node_segment_id()
        node_segment_node_filter = self._get_yp_node_segment_node_filter(node_segment_id)
        if self._restrict_nodes_to_specified_in_podset:
            podset_node_filter = self._get_yp_podset_node_filter()
            return " AND ".join(
                _filter
                for _filter in [node_segment_node_filter, podset_node_filter]
                if _filter
            )
        return node_segment_node_filter

    def _fetch_yp_nodes_resources(self, yp_nodes):
        # NOTE(rocco66): similar to _fetch_nodes_disks_storage_class, needs refactoring
        yp_nodes_set = set(yp_nodes)
        result = defaultdict(set)
        for node_id, disk_storage_class in self.select_objects(
            "resource",
            filter="[/meta/kind]='disk'",
            selectors=["/meta/node_id", "/spec/disk/storage_class"],
        ):
            if node_id in yp_nodes_set:
                result[node_id].add(disk_storage_class)
        return result

    def fetch_yp_nodes(self, selectors: typing.List[str] = None) -> typing.List[tuple]:
        if not selectors:
            selectors = ["/meta/id"]
        node_filter = self._get_yp_node_filter()
        result = list(self.select_objects(
            "node",
            filter=node_filter,
            selectors=selectors,
        ))
        self._logger.debug("Loaded '%d' nodes from YP API, filter='%s'" % (len(result), node_filter))
        return result

    def _fetch_nodes_disks_storage_class(self, nodes: typing.List[str]) -> typing.Dict[str, typing.Set[str]]:
        valid_storage_classes = ["hdd", "ssd"]

        nodes_as_string = ", ".join(f'"{node}"' for node in nodes)
        nodes_map = collections.defaultdict(set)

        node_spec_pairs = self.select_objects(
            "resource",
            filter=f"[/meta/node_id] IN ({nodes_as_string})",
            selectors=["/meta/node_id", "/spec/disk/storage_class"],
        )
        for node_id, disk_storage_class in node_spec_pairs:
            if not disk_storage_class:
                # TODO: Need info from YP how these specs work.
                # https://paste.yandex-team.ru/1091865 Empty resource's '/spec/disk' specs are normal?
                self._logger.debug(("Node '%s' does not have 'storage_class' in one of its '/spec/disk' "
                                    "resource") % node_id)
                continue
            if disk_storage_class not in valid_storage_classes:
                raise YpLitePodsManagerException(("Node '%s' have unknown 'storage_class' '%s' in one of its "
                                                  "'/spec/disk' resource") % (node_id, disk_storage_class))
            nodes_map[node_id].add(disk_storage_class)
        return dict(nodes_map)

    def raw_fetch_pods(self, selectors):
        return self.select_objects(
            "pod",
            filter=f'[/meta/pod_set_id]="{self.get_podset_name()}"',
            selectors=selectors,
        )

    def _fetch_pods(self) -> typing.Dict[str, PodInfo]:
        pods = {}
        raw_pods = self.raw_fetch_pods(selectors=[
            "/meta/id",
            "/status/dns/persistent_fqdn",
            "/status/scheduling/node_id",
            "/spec/scheduling/hints",
            "/spec/disk_volume_requests",
        ])
        for pod_id, fqdn, node_id, scheduling_hints, disk_requests in raw_pods:
            scheduling_hint_node_id = None
            if scheduling_hints:
                scheduling_hint_node_id = scheduling_hints[0]["node_id"]

            storage_class = None
            using_storage_classes = {r["storage_class"] for r in disk_requests}
            if len(using_storage_classes) == 1:
                storage_class_str = next(iter(using_storage_classes))
                if storage_class_str == "hdd":
                    storage_class = PodDiskReq.PodDiskReqHdd
                elif storage_class_str == "ssd":
                    storage_class = PodDiskReq.PodDiskReqSsd
                else:
                    self._logger.error("Unknown disk req '%s' (pod %s)", storage_class_str, pod_id)
            else:
                self._logger.error("Several disk reqs '%s' (pod %s)", using_storage_classes, pod_id)

            pods[pod_id] = PodInfo(fqdn, node_id, scheduling_hint_node_id, storage_class)

        self._logger.debug("Loaded '%d' pods from YP API" % len(pods))
        return pods

    def _fetch_nanny_instances(self) -> typing.List[str]:
        with stat_server.nanny_timing():
            active_runtime_attrs = self._nanny_service_repo_client.get_runtime_attrs(self._nanny_service_name)
        nanny_instances = [
            pod["pod_id"]
            for pod in active_runtime_attrs["content"]["instances"]["yp_pod_ids"]["pods"]
        ]
        self._logger.debug("Loaded '%d' Nanny instances from Nanny API" % len(nanny_instances))
        return nanny_instances

    @staticmethod
    def _fill_node_resources(node_id, yp_nodes_resources, pod_data):
        node_resources = yp_nodes_resources.get(node_id)
        if node_resources:
            pod_data.yp_node_has_hdd = "hdd" in node_resources
            pod_data.yp_node_has_ssd = "ssd" in node_resources

    @staticmethod
    def _merge(yp_nodes,
               yp_nodes_resources,
               pods: typing.Dict[str, PodInfo],
               nanny_instances) -> typing.List[YpLitePodsManagerPodData]:
        actual_yp_nodes_set = {n for n in yp_nodes}
        yp_nodes_with_pod = set()
        rows = []

        for _id, pod_info in pods.items():
            pod_data = YpLitePodsManagerPodData(
                yp_pod_id=_id,
                yp_pod_fqdn=pod_info.fqdn,
                scheduling_hint_node_id=pod_info.scheduling_hint_node_id,
                yp_pod_disk_req=pod_info.yp_pod_disk_req,
            )

            # If pod is scheduled on some node...
            node_id = pod_info.node_id
            if node_id:
                yp_nodes_with_pod.add(node_id)
                pod_data.yp_node_id = node_id
                if node_id in actual_yp_nodes_set:
                    pod_data.yp_node_matches_node_filter = True
                    YpLitePodsManager._fill_node_resources(node_id, yp_nodes_resources, pod_data)
                    if node_info := yp_nodes.get(node_id):
                        pod_data.yp_node_hfsm_state = node_info.hfsm_state

            # Try to remove pod from Nanny instances list.
            try:
                nanny_instances.remove(_id)
                pod_data.nanny_instance_pod_id = _id
            except ValueError:
                pass

            rows.append(pod_data)

        # Add YP nodes leftovers.
        for node_id in set(actual_yp_nodes_set) - yp_nodes_with_pod:
            pod_data = YpLitePodsManagerPodData(yp_node_id=node_id, yp_node_matches_node_filter=True)
            YpLitePodsManager._fill_node_resources(node_id, yp_nodes_resources, pod_data)
            if node_info := yp_nodes.get(node_id):
                pod_data.yp_node_hfsm_state = node_info.hfsm_state
            rows.append(pod_data)

        # Add Nanny instances leftovers.
        for _instance in nanny_instances:
            rows.append(YpLitePodsManagerPodData(nanny_instance_pod_id=_instance))

        return rows

    @staticmethod
    def _row_matcher_pods_seen(row: YpLitePodsManagerPodData) -> bool:
        return row.yp_pod_id is not None

    @staticmethod
    def _row_matcher_nodes_seen(row: YpLitePodsManagerPodData) -> bool:
        return row.yp_node_id is not None

    @staticmethod
    def _row_matcher_nodes_up_seen(row: YpLitePodsManagerPodData) -> bool:
        return row.yp_node_id is not None and row.yp_node_hfsm_state == "up"

    @staticmethod
    def _row_matcher_nanny_instances_seen(row: YpLitePodsManagerPodData) -> bool:
        return row.nanny_instance_pod_id is not None

    @staticmethod
    def _row_matcher_pods_actual_allocated(row: YpLitePodsManagerPodData) -> bool:
        return row.yp_pod_id is not None and row.yp_node_id is not None and row.yp_node_matches_node_filter

    @staticmethod
    def _row_matcher_pods_not_allocated(row: YpLitePodsManagerPodData) -> bool:
        return row.yp_pod_id is not None and row.yp_node_id is None

    @staticmethod
    def _row_matcher_pods_in_deploy_system(row: YpLitePodsManagerPodData) -> bool:
        return row.yp_pod_id is not None and row.nanny_instance_pod_id is not None

    @staticmethod
    def _row_matcher_pods_not_in_deploy_system(row: YpLitePodsManagerPodData) -> bool:
        return row.yp_pod_id is not None and row.nanny_instance_pod_id is None

    @staticmethod
    def _row_matcher_pods_allocated_and_in_deploy_system(row: YpLitePodsManagerPodData) -> bool:
        return row.yp_pod_id is not None and row.yp_node_id is not None and row.nanny_instance_pod_id is not None

    @staticmethod
    def _row_matcher_pods_allocated_on_non_matched_node(row: YpLitePodsManagerPodData) -> bool:
        return row.yp_pod_id is not None and row.yp_node_id is not None and not row.yp_node_matches_node_filter

    @staticmethod
    def _row_matcher_pods_not_allocated_and_not_in_deploy_system(row: YpLitePodsManagerPodData) -> bool:
        return row.yp_pod_id is not None and row.yp_node_id is None and row.nanny_instance_pod_id is None

    @staticmethod
    def _row_matcher_nodes_without_pods(row: YpLitePodsManagerPodData) -> bool:
        return row.yp_node_id is not None and row.yp_pod_id is None

    @staticmethod
    def _row_matcher_nanny_instances_without_pods(row: YpLitePodsManagerPodData) -> bool:
        return row.nanny_instance_pod_id is not None and row.yp_pod_id is None

    def select_objects(self, *args, **kwargs):
        if "batching_options" not in kwargs:
            kwargs["batching_options"] = BatchingOptions(attempt_count=3)
        with stat_server.yp_timing():
            return self._yp_client.select_objects(*args, **kwargs)

    def _wait_for_ts(self, ts_to_wait_for: int = 0, max_tries: int = 18, sleep_between_tries: int = 5):
        # TODO(rocco66): single _wait_for_ts function
        # TODO(rocco66): time offset instead of large tries count
        table_name = YpLitePods.table_name()

        def _read_max_ts() -> int:
            max_ts_query = f"""
                (
                    SELECT toUnixTimestamp(MAX(ts)) AS max_ts
                    FROM {table_name}
                    WHERE (nanny_service_name = '{self._nanny_service_name}')
                )
            """
            row = next(self._clickhouse_client.select(max_ts_query))
            return row.max_ts

        max_ts = _read_max_ts()
        ts_present = max_ts == ts_to_wait_for
        num_tries = 0
        while not ts_present and num_tries < max_tries:
            self._logger.debug("Polling for timestamp '%d', current MAX(ts) is '%d'..." % (ts_to_wait_for, max_ts))
            time.sleep(sleep_between_tries)
            num_tries += 1
            max_ts = _read_max_ts()
            if max_ts > ts_to_wait_for:
                self._logger.info("No current minute's timestamp '%r' found, got more recent MAX(ts)=='%r'" %
                                  (ts_to_wait_for, max_ts))
                raise YpLitePodsManagerException("No current minute's timestamp found, got more recent one")
            ts_present = max_ts == ts_to_wait_for
        if not ts_present:
            self._logger.info(("Have been waiting too long for current minute's timestamp '%r', current MAX(ts)"
                               "=='%r'") % (ts_to_wait_for, max_ts))
            raise YpLitePodsManagerException("Have been waiting too long for current minute's timestamp")
