#!/usr/bin/env python2

import argparse
import datetime
import json
import os
import time

import requests


JUGGLER_URL = "http://juggler-api.search.yandex.net/v2/history/get_check_history"
JUGGLER_HEADERS = {
    "accept": "application/json",
    "Content-Type": "application/json"
}
STAT_BETA_FQDN = "upload.stat-beta.yandex-team.ru"
STAT_FQDN = "upload.stat.yandex-team.ru"
STAT_OAUTH_TOKEN = None
STAT_REPORT = {
    "name": "Yandex/Infra/RtcSlaTentacles/Metrics",
    "title": "RTC SLA Tentacles availability metrics",
    "scale": "d",
    "report_config": {
        "dimensions": [
            {"fielddate": "date"},
            {"data": "tree"}
        ],
        "measures": [
            {"value": "number"},
        ],
        "graphs": [
            {"fields": "value"},
        ]
    }
}
STAT_OAUTH_TOKEN_BELONGS_TO_USER = "robot-tentacles"
TENTACLES_ALL_DCS = ["sas", "man", "vla", "msk", "yp_sas", "yp_man", "yp_vla", "yp_man_pre"]


def get_page(post_json):
    request = requests.post(url=JUGGLER_URL,
                            headers=JUGGLER_HEADERS,
                            json=post_json)
    if request.status_code != 200:
        return False
    return request.json()


def get_history_from_juggler(service_name,
                             start_time=1000000000,
                             end_time=2000000000,
                             update_cache=True):

    if not isinstance(start_time, int) or not isinstance(end_time, int):
        raise ValueError("start_time and end_time must be of type int")

    post_json = {
        "host": "rtc_sla_cluster_health",
        "statuses": ["CRIT", "OK"],
        "service": service_name,
        "since": start_time,
        "page_size": 200,
        "until": end_time,
        "page": 1
    }
    all_pages = []

    previous_status = None

    while True:
        page_data = get_page(post_json)
        if not page_data:
            return False

        for state in page_data["states"]:
            state_data = {}
            state_data["status"] = state["status"]
            # Juggler check description changes here
            # state_data["status_time"] = datetime.fromtimestamp(int(state["status_time"])).strftime('%Y-%m-%d %H:%M:%S')
            state_data["status_time"] = int(state["status_time"])
            if state_data["status"] != previous_status:
                # New or changed
                all_pages.append(state_data)
                previous_status = state_data["status"]
            else:
                continue
        if page_data["has_more"]:
            post_json["page"] += 1
        else:
            break

    history_data = list(reversed(all_pages))

    # history_data is a list of dicts and is sorted by time

    if update_cache:
        write_history_to_cache_file(history_data, service_name + "_history.json")

    return history_data


def write_history_to_cache_file(data, cache_filename):
    with open(cache_filename, "w") as file_descriptor:
        json.dump(data, file_descriptor)


def get_history_from_cache_file(cache_filename):
    with open(cache_filename) as file_descriptor:
        return json.load(file_descriptor)


def load_history(service_name, source):
    if source == "juggler":
        history = get_history_from_juggler(service_name)
    elif source == "cache":
        history = get_history_from_cache_file(cache_filename=service_name + "_history.json")
    else:
        raise ValueError("Reading history from {} is not implemented".format(source))

    list_of_periods = []

    # Construct first virtual period
    first_period = {
        "start_time": 0,
        "end_time": history[0]["status_time"],
        "duration": None,
        "status": None
    }
    list_of_periods.append(first_period)

    # Append middle periods
    point_counter = 0
    for point in history:
        if point_counter == 0: # First, oldest page
            prev_point = point
            point_counter += 1
            continue
        period = {
            "start_time": prev_point["status_time"],
            "end_time": point["status_time"],
            "duration": point["status_time"] - prev_point["status_time"],
            "status": prev_point["status"]
        }
        list_of_periods.append(period)
        prev_point = point
        point_counter += 1

    # Construct last virtual period
    last_period = {
        "start_time": history[-1]["status_time"],
        "end_time": 2000000000,
        "duration": None,
        "status": history[-1]["status"]
    }
    list_of_periods.append(last_period)

    return list_of_periods



def get_period_index_at_specific_time(juggler_history, unixtime):
    i = 0
    for period in juggler_history:
        if period["start_time"] <= unixtime < period["end_time"]:
            return i
        i += 1
    raise ValueError("Can not find period's index at given time ", unixtime, "in history")



def get_timeline(juggler_history, start_time=None, end_time=None):
    timeline = []

    history_start_time = juggler_history[1]["start_time"]

    # Default period is whole history.
    if not start_time:
        start_time = history_start_time
    if not end_time:
        end_time = int(time.time())

    if end_time <= start_time:
        raise ValueError("Timeline's end_time {} is equal or less than start_time {}".format(end_time, start_time))


    start_period_index = get_period_index_at_specific_time(juggler_history, start_time)
    end_period_index = get_period_index_at_specific_time(juggler_history, end_time)

    if end_period_index < start_period_index:
        raise ValueError("End period index is less than start period index, maybe history is not sorted by time")

    timeline_item = {}
    if start_period_index == end_period_index:
        timeline_item = {
            "start_time": start_time,
            "end_time": end_time,
            "status": juggler_history[start_period_index]["status"],
            "duration": end_time - start_time
        }
        timeline.append(timeline_item)
        return timeline

    # start_period_index < end_period_index
    i = 0
    for period in juggler_history:
        if i == start_period_index:
            timeline_item = {
                "start_time": start_time,
                "end_time": period["end_time"],
                "status": period["status"],
                "duration": period["end_time"] - start_time
            }
            timeline.append(timeline_item)

        elif start_period_index < i < end_period_index:
            timeline.append(period)

        elif i == end_period_index:
            timeline_item = {
                "start_time": period["start_time"],
                "end_time": end_time,
                "status": period["status"],
                "duration": end_time - period["start_time"]
            }
            timeline.append(timeline_item)
            break

        i += 1

    return timeline


def get_timeline_mean_times(timeline):
    """
        http://blog.fosketts.net/2011/07/06/defining-failure-mttr-mttf-mtbf/

        Mean Time to Failure (MTTF) - mean period of "OK" = whole "OK" time / number of periods of "OK"
        Mean Time to Repair (MTTR) - mean period of "CRIT" = whole "CRIT" time / number of periods of "CRIT"
        Mean Time Between Failures (MTBF) = MTTF + MTTR
    """
    mean_times = {
        "mttf": 0.0, # Used only to calculate MTBF, deleted later
        "mttr": 0.0,
        "mtbf": 0.0,
    }

    number_of_ok_periods = 0
    number_of_crit_periods = 0

    total_ok_time = 0
    total_crit_time = 0


    for period in timeline:
        if period["status"] == "OK":
            number_of_ok_periods += 1
            total_ok_time += period["duration"]
        elif period["status"] == "CRIT":
            number_of_crit_periods += 1
            total_crit_time += period["duration"]
        elif period["status"] is None:
            continue

    if number_of_ok_periods > 0:
        mean_times["mttf"] = total_ok_time / number_of_ok_periods
    else:
        mean_times["mttf"] = 0
    if number_of_crit_periods > 0:
        mean_times["mttr"] = total_crit_time / number_of_crit_periods
    else:
        mean_times["mttr"] = 0
    mean_times["mtbf"] = mean_times["mttf"] + mean_times["mttr"]

    # MTTF is not needed
    del mean_times["mttf"]

    return mean_times


def get_time_spent_in_status_percent(timeline):
    statuses_percent = {
        "ok": 0.0,
        "crit": 0.0
    }

    duration_ok = 0
    duration_crit = 0

    for period in timeline:
        if period["status"] == "OK":
            duration_ok += period["duration"]
        elif period["status"] == "CRIT":
            duration_crit += period["duration"]
        elif period["status"] is None:
            continue

    sum_of_durations = duration_ok + duration_crit
    if sum_of_durations > 0:
        statuses_percent["ok"] = duration_ok / float(sum_of_durations) * 100
        statuses_percent["crit"] = duration_crit / float(sum_of_durations) * 100
    else:
        statuses_percent["ok"] = 0
        statuses_percent["crit"] = 0

    return statuses_percent


def number_of_seconds_to_human(sec):
    int_sec = int(sec)
    if int_sec < 60:
        return "%d seconds" % int_sec
    elif int_sec < 60*60:
        return "%.1f minutes" % (float(int_sec) / 60)
    elif int_sec < 60*60*24:
        return "%.1f hours" % (float(int_sec) / (60*60))
    return "%.1f days" % (float(int_sec) / (60*60*24))


def print_timeline(timeline):
    row_format = "{:>25}" * 5
    print(row_format.format("Start time", "Status", "Duration, seconds", "Duration", "End time"))
    for period in timeline:
        print(row_format.format(
            datetime.datetime.fromtimestamp(period["start_time"]).strftime('%Y-%m-%d %H:%M:%S'),
            period["status"],
            period["duration"],
            number_of_seconds_to_human(period["duration"]),
            datetime.datetime.fromtimestamp(period["end_time"]).strftime('%Y-%m-%d %H:%M:%S')
            ))


def parse_date(date_as_string):
    """ Returns date object from string. """
    return datetime.datetime.strptime(date_as_string, "%Y-%m-%d").date()


def create_stat_report(date, mean_times, time_spent_in_status_percent):
    # {"vla": {"mttr": 0, "mtbf": 111998, ...},
    # to
    # {
    #   "values": [
    #     {
    #         "fielddate": "2018-08-29",
    #         "data": "\tsas\tmttr_min\t",
    #         "value": 42
    #     },
    #     {
    #         "fielddate": "2018-08-29",
    #         "data": "\tsas\tmtbf_day\t",
    #         "value": 24
    #     },
    #     ....
    #    ]
    # }

    report_json = {}
    report_values = []

    # Raw values are in seconds, we scale them here to minutes, hours and dates.
    # Correponding, most suitable scaled signal is to be chosen when creating a chart.
    scales = {
        "min": 60,
        "hour": 60*60,
        "day": 60*60*24
    }

    # Append scaled MTTR and MTBF
    for dc in mean_times.keys():
        for signal in mean_times[dc].keys():
            for scale_name, scale_divisor in scales.items():
                value_item = {}
                value_item["fielddate"] = date
                value_item["data"] = "\t{dc}\t{signal}_{scale}\t".format(dc=dc, signal=signal, scale=scale_name)
                value_item["value"] = float(mean_times[dc][signal]) / scale_divisor
                report_values.append(value_item)

    # Append percent of time in CRIT and OK
    for dc in time_spent_in_status_percent.keys():
        for signal in time_spent_in_status_percent[dc].keys():
            value_item = {}
            value_item["fielddate"] = date
            value_item["data"] = "\t{dc}\tpercent_of_time_in_{signal}\t".format(dc=dc, signal=signal)
            value_item["value"] = time_spent_in_status_percent[dc][signal]
            report_values.append(value_item)

    report_json["values"] = report_values
    return report_json


def upload_report_to_stat(stat_fqdn, stat_oauth_token, data):
    """ Creates/uploads report to stat. """

    # upload_to_stat(STAT_BETA_FQDN, STAT_OAUTH_TOKEN, report_json)
    # Importing requests here because there is no numpy in YT runtime environment.

    # Will be returned by this function
    upload_result = {
        "if_success": None,
        "message": None
    }

    request_headers = {
        "Authorization": "OAuth {}".format(stat_oauth_token)
    }
    who_am_i_url = "https://{}/_v3/whoami/".format(stat_fqdn)
    report_config_url = "https://{}/_api/report/config".format(stat_fqdn)
    report_data_url = "https://{}/_api/report/data".format(stat_fqdn)

    # Check who am I
    resp = requests.get(who_am_i_url, headers=request_headers, verify=False)
    try:
        resp_json = json.loads(resp.text)

        # Check if OAuth token provided belongs to correct user
        if resp_json["username"] != STAT_OAUTH_TOKEN_BELONGS_TO_USER:
            upload_result["if_success"] = False
            upload_result["message"] = "OAuth token does not belong to user {}, won't try to \
    upload.".format(STAT_OAUTH_TOKEN_BELONGS_TO_USER)
            return upload_result
    except ValueError:
        upload_result["if_success"] = False
        upload_result["message"] = resp.text
        return upload_result

    # Upload report config
    resp = requests.post(
        report_config_url,
        headers=request_headers,
        data={
            "json_config": json.dumps(
                {
                    "user_config": STAT_REPORT["report_config"],
                    "title": STAT_REPORT["title"]
                }
            ),
            "name": STAT_REPORT["name"],
            "scale": STAT_REPORT["scale"]
        },
        verify=False
    )
    if not resp.ok:
        upload_result["if_success"] = False
        upload_result["message"] = resp.text
        return upload_result

    # Upload data
    resp = requests.post(
        report_data_url,
        headers=request_headers,
        data={
            "json_data": json.dumps(data),
            "name": STAT_REPORT["name"],
            "scale": STAT_REPORT["scale"]
        },
        verify=False
    )

    # Form function result
    if not resp.ok:
        upload_result["if_success"] = False
    else:
        upload_result["if_success"] = True
    upload_result["message"] = resp.text

    return upload_result


def parse_args():
    parser = argparse.ArgumentParser("RTC SLA Tentacles MTTR/MTBF/MTTF generator")
    parser.add_argument("--date",
                        help="End date in YYYY-MM-DD format. Will calculate metrics till that day. Defaults to yesterday",
                        type=parse_date,
                        default=datetime.date.today() - datetime.timedelta(days=1))
    parser.add_argument("--start-date",
                        help="(optional) Start date in YYYY-MM-DD format. Will calculate metrics from that day.",
                        type=parse_date,
                        default=None)
    parser.add_argument("--cache",
                        help="Use cached Juggler API results",
                        action="store_true")
    parser.add_argument("--print-results",
                        help="Print human-friendly results",
                        action="store_true")
    parser.add_argument("--stat-oauth-token",
                        help="Stat OAuth token for robot-tentacles, or STAT_OAUTH_TOKEN in env",
                        default=STAT_OAUTH_TOKEN)
    parser.add_argument("--upload-to-stat-beta",
                        help="Upload results to stat-beta.yandex-team.ru",
                        action="store_true")
    parser.add_argument("--upload-to-stat",
                        help="Upload results to stat.yandex-team.ru",
                        action="store_true")
    args = parser.parse_args()
    return args


if __name__ == "__main__":
    args = parse_args()

    if not args.stat_oauth_token:
        try:
            args.stat_oauth_token = os.environ["STAT_OAUTH_TOKEN"]
        except Exception:
            args.stat_oauth_token = None

    # Check if OAuth token for stat.yandex-team.ru provided by user
    if args.upload_to_stat_beta or args.upload_to_stat:
        if not args.stat_oauth_token:
            print("ERROR: No OAuth token provided. Token must belong \
to user {}".format(STAT_OAUTH_TOKEN_BELONGS_TO_USER))
            exit(1)

    if args.cache:
        history_source = "cache"
    else:
        history_source = "juggler"

    end_date_unixtime = int(time.mktime(args.date.timetuple()))
    end_date_human = str(args.date)
    if args.start_date:
        start_date_unixtime = int(time.mktime(args.start_date.timetuple()))
        start_date_human = str(args.start_date)
    else:
        start_date_unixtime = None
        start_date_human = None

    timelines = {}
    mean_times = {}
    time_spent_in_status_percent = {}
    for dc in TENTACLES_ALL_DCS:
        dc_juggler_history = load_history(service_name=dc, source=history_source)
        dc_timeline = get_timeline(dc_juggler_history,
                                   start_time=start_date_unixtime,
                                   end_time=end_date_unixtime)
        timelines[dc] = dc_timeline
        mean_times[dc] = get_timeline_mean_times(dc_timeline)
        time_spent_in_status_percent[dc] = get_time_spent_in_status_percent(dc_timeline)

    # Print
    if args.print_results:
        for dc in TENTACLES_ALL_DCS:
            print("Timeline for {}".format(dc))
            print_timeline(timelines[dc])
            print("Mean times: MTTR {}, MTBF {}".format(number_of_seconds_to_human(mean_times[dc]["mttr"]),
                                                        number_of_seconds_to_human(mean_times[dc]["mtbf"])))
            print("Percent of time in status: OK {0:.2f}, CRIT {1:.2f}".format(time_spent_in_status_percent[dc]["ok"],
                                                                               time_spent_in_status_percent[dc]["crit"]))

    if args.upload_to_stat_beta or args.upload_to_stat:
        report_json = create_stat_report(end_date_human, mean_times, time_spent_in_status_percent)

    if args.upload_to_stat_beta:
        stat_beta_response = upload_report_to_stat(STAT_BETA_FQDN, args.stat_oauth_token, report_json)
        print("Uploading to {} result: if_success={}, message={}".format(
            STAT_BETA_FQDN,
            stat_beta_response["if_success"],
            stat_beta_response["message"].encode("utf-8")))
        if not stat_beta_response["if_success"]:
            exit_code = 1

    if args.upload_to_stat:
        stat_response = upload_report_to_stat(STAT_FQDN, args.stat_oauth_token, report_json)
        print("Uploading to {} result: if_success={}, message={}".format(
            STAT_FQDN,
            stat_response["if_success"],
            stat_response["message"].encode("utf-8")))
        if not stat_response["if_success"]:
            exit_code = 1

    exit(0)
