import * as React from 'react';
import * as Enzyme from 'enzyme';
import * as Adapter from 'enzyme-adapter-react-16';

import { CalendarHeatmap } from './CalendarHeatmap';
import { dateNDaysAgo, shiftDate } from './CalendarHeatmap.helpers';
import { DAY_LABELS } from './CalendarHeatmap.constants';

const b = (el) => `.calendar-heatmap__${el}`;

const { shallow } = Enzyme;

Enzyme.configure({ adapter: new Adapter() });

const getWrapper = (
    overrideProps,
    renderMethod = 'shallow'
): Enzyme.ShallowWrapper => {
    const defaultProps = {
        values: [],
    };
    return Enzyme[renderMethod](
        <CalendarHeatmap {...defaultProps} {...overrideProps} />
    );
};

describe('CalendarHeatmap', () => {
    const values = [
        { date: new Date('2017-06-01') },
        { date: new Date('2017-06-02') },
        { date: new Date('2018-06-01') },
        { date: new Date('2018-06-02') },
        { date: new Date('2018-06-03') },
    ];

    it('should render as an svg', () => {
        const wrapper = shallow(<CalendarHeatmap values={[]} />);

        expect(wrapper.find('svg')).toHaveLength(1);
    });

    it('should not throw exceptions in base case', () => {
        expect(() => <CalendarHeatmap values={[]} />).not.toThrow();
    });

    it('shows values within its original date range', () => {
        const wrapper = shallow(
            <CalendarHeatmap
                endDate={new Date('2017-12-31')}
                startDate={new Date('2017-01-01')}
                values={values}
            />
        );

        expect(wrapper.find(b('cell_filled')).length).toBe(2);
    });

    it('should handle string formatted date range', () => {
        const wrapper = shallow(
            <CalendarHeatmap
                endDate="2017-12-31"
                startDate="2017-01-01"
                values={values}
            />
        );

        expect(wrapper.find(b('cell_filled')).length).toBe(2);
    });
});

describe('CalendarHeatmap props', () => {
    it('values', () => {
        const values = [
            { date: '2016-01-01' },
            { date: new Date('2016-01-02').getTime() },
            { date: new Date('2016-01-03') },
        ];
        const wrapper = shallow(
            <CalendarHeatmap
                endDate={new Date('2016-02-01')}
                startDate={new Date('2015-12-20')}
                values={values}
            />
        );

        expect(wrapper.find(b('cell_filled'))).toHaveLength(values.length);
    });

    it('startDate', () => {
        const today = new Date();
        const wrapper = shallow<CalendarHeatmap>(
            <CalendarHeatmap values={[]} endDate={today} startDate={today} />
        );

        expect(
            today.getDate() === wrapper.instance().getEndDate().getDate() &&
                today.getMonth() === wrapper.instance().getEndDate().getMonth()
        ).toBe(true);
    });

    it('endDate', () => {
        const today = new Date();
        const wrapper = shallow<CalendarHeatmap>(
            <CalendarHeatmap
                values={[]}
                endDate={today}
                startDate={dateNDaysAgo(10)}
            />
        );

        expect(
            today.getDate() === wrapper.instance().getEndDate().getDate() &&
                today.getMonth() === wrapper.instance().getEndDate().getMonth()
        ).toBe(true);
    });

    it('showMonthLabels', () => {
        const visible = shallow(
            <CalendarHeatmap
                startDate={dateNDaysAgo(100)}
                values={[]}
                showMonthLabels
            />
        );

        expect(visible.find('text').length).toBeGreaterThan(0);

        const hidden = shallow(
            <CalendarHeatmap values={[]} showMonthLabels={false} />
        );

        expect(hidden.find('text')).toHaveLength(0);
    });

    it('showWeekdayLabels', () => {
        const visible = shallow(
            <CalendarHeatmap
                startDate={dateNDaysAgo(7)}
                values={[]}
                showWeekdayLabels
            />
        );

        expect(visible.find('text').length).toBeGreaterThan(2);

        const hidden = shallow(
            <CalendarHeatmap
                values={[]}
                showMonthLabels={false}
                showWeekdayLabels={false}
            />
        );

        expect(hidden.find('text')).toHaveLength(0);

        const vertical = shallow(
            <CalendarHeatmap values={[]} showWeekdayLabels />
        );
        const weekdayLabelsLength = DAY_LABELS.filter(Boolean).length;

        expect(vertical.find(b('weekday-label_visible'))).toHaveLength(
            weekdayLabelsLength
        );
    });

    it('transformDayElement', () => {
        const transform = (rect) =>
            React.cloneElement(rect, { 'data-test': 'ok' });
        const today = new Date();
        const expectedStartDate = shiftDate(today, -1);
        const wrapper = shallow(
            <CalendarHeatmap
                values={[{ date: expectedStartDate }, { date: today }]}
                endDate={today}
                startDate={expectedStartDate}
                transformDayElement={transform}
            />
        );

        expect(wrapper.find('[data-test="ok"]').length).toBeGreaterThan(0);
    });

    describe('tooltipDataAttrs', () => {
        it('allows a function to be passed', () => {
            const today = new Date();
            const numDays = 10;
            const expectedStartDate = shiftDate(today, -numDays + 1);
            const wrapper = shallow(
                <CalendarHeatmap
                    values={[
                        { date: today, count: 1 },
                        { date: expectedStartDate, count: 0 },
                    ]}
                    endDate={today}
                    startDate={expectedStartDate}
                    tooltipDataAttrs={({ count }) => ({
                        'data-tooltip': `Count: ${count}`,
                    })}
                />
            );

            expect(wrapper.find('[data-tooltip="Count: 1"]')).toHaveLength(1);
        });
    });

    describe('event handlers', () => {
        const count = 999;
        const startDate = '2018-06-01';
        const endDate = '2018-06-03';
        const values = [{ date: '2018-06-02', count }];
        const props = {
            values,
            startDate,
            endDate,
        };
        const [expectedValue] = values;

        it('calls props.onClick with the correct value', () => {
            const onClick = jest.fn();
            const wrapper = getWrapper({ ...props, onClick });

            const rect = wrapper.find(b('cell_filled'));
            rect.simulate('click');

            expect(onClick).toHaveBeenCalledWith(expectedValue);
        });
    });
});
