import React from 'react';
// import block from 'bem-cn-lite';
import {isEmpty, isEqual} from 'lodash';
// import Tooltip from 'react-tooltip';
import classNames from 'classnames';

import I18n from './i18n';
import styles from './CalendarHeatmap.module.css';

import {
    DAY_LABELS,
    DAYS_IN_WEEK,
    LEGEND_COLORS,
    MILLISECONDS_IN_ONE_DAY,
    MONTH_LABELS,
    OFFSET_FROM_TOP,
    WEEKDAY_LABEL_SIZE,
} from './CalendarHeatmap.constants';
import {convertToDate, dateNDaysAgo, getBeginningTimeForDate, getRange, shiftDate} from './CalendarHeatmap.helpers';
import {CommitCounts, HeatmapCache, TooltipDataAttrs} from './CalendarHeatmap.types';

const SQUARE_SIZE = 11;
const MONTH_LABEL_GUTTER_SIZE = 10;

// const b = block('calendar-heatmap');
const i18n = I18n.keyset('arcanum.heatmap');

export interface CalendarHeatmapProps {
    values: CommitCounts[];
    gutterSize: number; // Размер отступов между ячейками
    startDate: string | number | Date; // Дата начала видимого отрезка хитмапа
    monthLabels: string[];
    weekdayLabels: string[];
    endDate: string | number | Date; // Дата конца видимого отрезка хитмапа
    className?: string;
    showMonthLabels?: boolean;
    showWeekdayLabels?: boolean;
    showOutOfRangeDays?: boolean;
    tooltipDataAttrs?: TooltipDataAttrs | ((value: CommitCounts) => TooltipDataAttrs);
    titleForValue?: (value: CommitCounts | null) => string;
    classForValue?: (value: CommitCounts) => string;
    onClick?: Function;
    isLoading?: boolean;
    transformDayElement?: (rect: JSX.Element, value: CommitCounts | null, index: number) => JSX.Element; // Функция для трансформации svg элемента ячейки
}

interface CalendarHeatmapState {
    valueCache: HeatmapCache;
}

export class CalendarHeatmap extends React.Component<CalendarHeatmapProps, CalendarHeatmapState> {
    static defaultProps = {
        startDate: dateNDaysAgo(200),
        endDate: new Date(),
        gutterSize: 1,
        showMonthLabels: true,
        showWeekdayLabels: false,
        showOutOfRangeDays: false,
        monthLabels: MONTH_LABELS,
        weekdayLabels: DAY_LABELS,
    };

    state: Readonly<CalendarHeatmapState> = {
        valueCache: this.getValueCache(this.props.values),
    };

    componentDidUpdate(prevProps: CalendarHeatmapProps): void {
        if (!isEqual(this.props.values, prevProps.values)) {
            this.setState({
                valueCache: this.getValueCache(this.props.values),
            });
        }
    }

    getValueCache(values: CommitCounts[]): HeatmapCache {
        return values.reduce((memo, value) => {
            const date = convertToDate(value.date);
            const index = Math.floor(
                (Number(date) - Number(this.getStartDateWithEmptyDays())) / MILLISECONDS_IN_ONE_DAY
            );

            memo[index] = {
                value,
                className: this.getClassForCell(value),
                title: this.props.titleForValue ? this.props.titleForValue(value) : null,
                tooltipDataAttrs: this.getTooltipDataAttrsForValue(value),
            };

            return memo;
        }, {} as HeatmapCache);
    }

    getDateDifferenceInDays(): number {
        const {startDate} = this.props;

        const timeDiff = Number(this.getEndDate()) - Number(convertToDate(startDate));

        return Math.ceil(timeDiff / MILLISECONDS_IN_ONE_DAY);
    }

    getSquareSizeWithGutter(): number {
        return SQUARE_SIZE + this.props.gutterSize;
    }

    getMonthLabelSize(): number {
        if (!this.props.showMonthLabels) {
            return 0;
        }

        return SQUARE_SIZE + MONTH_LABEL_GUTTER_SIZE;
    }

    getWeekdayLabelSize(): number {
        if (!this.props.showWeekdayLabels) {
            return 0;
        }

        return WEEKDAY_LABEL_SIZE;
    }

    getStartDate(): Date {
        return shiftDate(this.getEndDate(), -this.getDateDifferenceInDays() + 1); // +1 тк мы считаем включительно по endDate
    }

    getEndDate(): Date {
        return getBeginningTimeForDate(convertToDate(this.props.endDate));
    }

    getStartDateWithEmptyDays(): Date {
        return shiftDate(this.getStartDate(), -this.getNumEmptyDaysAtStart());
    }

    getNumEmptyDaysAtStart(): number {
        return (this.getStartDate().getDay() || 7) - 1;
    }

    getNumEmptyDaysAtEnd(): number {
        const dayIndex = DAYS_IN_WEEK - 1;
        return dayIndex - (this.getEndDate().getDay() || 7) - 1;
    }

    getWeekCount(): number {
        const numDaysRoundedToWeek =
            this.getDateDifferenceInDays() + this.getNumEmptyDaysAtStart() + this.getNumEmptyDaysAtEnd();
        return Math.ceil(numDaysRoundedToWeek / DAYS_IN_WEEK);
    }

    getWeekWidth(): number {
        const countOfWeek = getRange(this.getWeekCount() - 1).length;

        return countOfWeek * this.getSquareSizeWithGutter();
    }

    getWidth(): number {
        return (
            this.getWeekCount() * this.getSquareSizeWithGutter() - (this.props.gutterSize - this.getWeekdayLabelSize())
        );
    }

    getHeight(): number {
        const {weekdayLabels, gutterSize} = this.props;

        const weekWidth = weekdayLabels.length * this.getSquareSizeWithGutter();

        return weekWidth + (this.getMonthLabelSize() - gutterSize);
    }

    getValueForIndex(index: number): CommitCounts | null {
        return this.state.valueCache[index]?.value;
    }

    getClassNameForIndex(index: number): string {
        const {valueCache} = this.state;

        if (valueCache[index]) {
            return valueCache[index].className;
        }

        return this.getClassForCell(null);
    }

    getTitleForIndex(index: number): string | null {
        const {titleForValue} = this.props;
        const {valueCache} = this.state;

        if (valueCache[index]) {
            return valueCache[index].title;
        }

        return titleForValue ? titleForValue(null) : null;
    }

    getTooltipDataAttrsForIndex(index: number): TooltipDataAttrs | undefined {
        const {isLoading} = this.props;
        const {valueCache} = this.state;

        if (isLoading) return;

        if (valueCache[index]) {
            return valueCache[index].tooltipDataAttrs;
        }

        return this.getTooltipDataAttrsForValue({
            date: undefined,
            count: undefined,
        });
    }

    getTooltipDataAttrsForValue(value: CommitCounts): TooltipDataAttrs | undefined {
        const {tooltipDataAttrs} = this.props;

        if (typeof tooltipDataAttrs === 'function') {
            return tooltipDataAttrs(value);
        }

        return tooltipDataAttrs;
    }

    getTransformForWeek(weekIndex: number): string {
        return `translate(${weekIndex * this.getSquareSizeWithGutter()}, 0)`;
    }

    getTransformForWeekdayLabels(): string {
        const x = this.getWeekWidth() + this.getMonthLabelSize();
        const y = SQUARE_SIZE + OFFSET_FROM_TOP;

        return `translate(${x}, ${y})`;
    }

    getTransformForAllWeeks(): string {
        return `translate(${0}, ${this.getMonthLabelSize()})`;
    }

    getViewBox(): string {
        return `0 0 ${this.getWidth()} ${this.getHeight()}`;
    }

    getSquareCoordinates(dayIndex: number): [number, number] {
        return [0, dayIndex * this.getSquareSizeWithGutter()];
    }

    getWeekdayLabelCoordinates(dayIndex: number): [number, number] {
        const {gutterSize} = this.props;

        const y = (dayIndex + 1) * SQUARE_SIZE + dayIndex * gutterSize;

        return [0, y];
    }

    getMonthLabelCoordinates(weekIndex: number): [number, number] {
        return [(weekIndex + 1) * this.getSquareSizeWithGutter(), 10];
    }

    getClassForCell(value: CommitCounts | null): string {
        const {isLoading} = this.props;

        if (isLoading) {
            return 'loading';
        }

        if (isEmpty(value) || !value?.count) {
            return 'empty';
        }

        if (value.count > 3) {
            return 'scale-4';
        }

        if (value.count > 2) {
            return 'scale-3';
        }

        if (value.count > 1) {
            return 'scale-2';
        }

        if (value.count > 0) {
            return 'scale-1';
        }

        return 'scale-0';
    }

    getTooltipLegendAttrs(title: string): object | undefined {
        return {'data-tip': title};
    }

    handleClick(value: CommitCounts | null): void {
        if (this.props.onClick) {
            this.props.onClick(value);
        }
    }

    render() {
        const {className} = this.props;

        return (
            <div className={styles.calendarHeatmap}>
                {/*<Tooltip delayHide={400} />*/}

                <svg className={styles.grid} viewBox={this.getViewBox()}>
                    <g className={''}>{this.renderMonthLabels()}</g>
                    <g transform={this.getTransformForAllWeeks()} className={''}>
                        {this.renderAllWeeks()}
                    </g>
                    <g transform={this.getTransformForWeekdayLabels()} className={''}>
                        {this.renderWeekdayLabels()}
                    </g>
                </svg>

{/*
                <div className={styles.legendBlock}>
                    <span>{i18n('label_heatmap-commits-count-title')}</span>
                    <ul className={styles.legend}>
                        {LEGEND_COLORS.map((item) => (
                            <li
                                key={item.colorClass}
                                className={classNames(styles.legendItem, styles[`color_${item.colorClass}`])}
                                {...this.getTooltipLegendAttrs(item.title)}
                            />
                        ))}
                    </ul>
                </div>
*/}
            </div>
        );
    }

    renderSquare(dayIndex: number, index: number): JSX.Element | null {
        const {transformDayElement, showOutOfRangeDays, isLoading} = this.props;

        const currentDayIndex = this.getNumEmptyDaysAtStart() + this.getDateDifferenceInDays();
        const indexOutOfRange = index >= currentDayIndex;

        if (indexOutOfRange && !showOutOfRangeDays) {
            return null;
        }

        const [x, y] = this.getSquareCoordinates(dayIndex);
        const value = this.getValueForIndex(index);

        const rect = (
            <rect
                key={index}
                width={SQUARE_SIZE}
                height={SQUARE_SIZE}
                x={x}
                y={y}
                className={classNames(
                    styles.cell,
                    styles[`color_${this.getClassNameForIndex(index)}`],
                    isLoading && styles.loading
                )}
                onClick={this.handleClick.bind(this, value)}
                {...this.getTooltipDataAttrsForIndex(index)}
            >
                <title>{this.getTitleForIndex(index)}</title>
            </rect>
        );

        return transformDayElement ? transformDayElement(rect, value, index) : rect;
    }

    renderWeek(weekIndex: number): JSX.Element {
        return (
            <g key={weekIndex} transform={this.getTransformForWeek(weekIndex)} className={''}>
                {getRange(DAYS_IN_WEEK).map((dayIndex) =>
                    this.renderSquare(dayIndex, weekIndex * DAYS_IN_WEEK + dayIndex)
                )}
            </g>
        );
    }

    renderAllWeeks(): JSX.Element[] {
        return getRange(this.getWeekCount()).map((weekIndex) => this.renderWeek(weekIndex));
    }

    renderMonthLabels() {
        const {showMonthLabels, monthLabels} = this.props;

        if (!showMonthLabels) {
            return null;
        }

        const weekRange = getRange(this.getWeekCount() - 1); // Не рендерим последний месяц, тк он будет обрезан

        return weekRange.map((weekIndex) => {
            const endOfWeek = shiftDate(this.getStartDateWithEmptyDays(), (weekIndex + 1) * DAYS_IN_WEEK - 1);

            const [x, y] = this.getMonthLabelCoordinates(weekIndex);

            return (
                endOfWeek.getDate() >= 1 &&
                endOfWeek.getDate() <= DAYS_IN_WEEK && (
                    <text key={weekIndex} x={x} y={y} className={''}>
                        {monthLabels![endOfWeek.getMonth()]}
                    </text>
                )
            );
        });
    }

    renderWeekdayLabels() {
        const {showWeekdayLabels, weekdayLabels} = this.props;

        if (!showWeekdayLabels) {
            return null;
        }

        return weekdayLabels.map((weekdayLabel, dayIndex) => {
            const [x, y] = this.getWeekdayLabelCoordinates(dayIndex);

            return (
                <text key={`${x}${y}`} x={x} y={y} className={classNames(dayIndex > 4 && styles.weekend)}>
                    {weekdayLabel}
                </text>
            );
        });
    }

    renderSkeletonLabel(x: number, y: number) {
        return <rect width={20} height={15} x={x} y={y} />;
    }
}
