import React from 'react';
import {AsideHeader, AsideHeaderMenuItem, AsideHeaderProps} from '@yandex-data-ui/navigation';
import {FontAwesomeIcon, FontAwesomeIconProps} from '@fortawesome/react-fontawesome';
import {faChartBar, faLineColumns} from '@fortawesome/pro-regular-svg-icons';
import {faTable} from '@fortawesome/pro-light-svg-icons';
import {Link, useHistory, useLocation} from 'react-router-dom';
import {ReactComponent as Tentacles} from './Tentacles.svg';
import {MenuUser} from './MenuUser';
import {IncidentIcon} from '../icons/IncidentIcon';
import routes from '../../routes';
import {LogoProps} from '@yandex-data-ui/navigation/build/components/AsideHeader/Logo/Logo';

type IconProps = Omit<FontAwesomeIconProps, 'icon'>;

interface Icon extends React.FunctionComponent<IconProps> {}

export const OverviewIcon: Icon = React.memo(() => <FontAwesomeIcon icon={faLineColumns} />);
export const SpiChartIcon: Icon = React.memo(() => <FontAwesomeIcon icon={faChartBar} />);
export const MissingNodesIcon: Icon = React.memo(() => <FontAwesomeIcon icon={faTable} />);
export const TentaclesIcon: Icon = React.memo(() => <Tentacles />);

const menuItemWrapper: AsideHeaderMenuItem['itemWrapper'] = (node, item) => {
    return (
        <Link to={String(item.link)} className={'link'} style={{color: 'var(--yc-color-text-primary)'}}>
            {node}
        </Link>
    );
};

interface NavigationProps {}

export const Navigation: React.FunctionComponent<NavigationProps> = (props) => {
    const {pathname} = useLocation();
    const history = useHistory();
    const toHome: LogoProps['onLogoIconClick'] = React.useCallback(
        (event) => {
            event.preventDefault();
            history.push(routes.home);
        },
        [history]
    );

    const menuItems = React.useMemo(() => {
        return [
            {
                id: 'slo',
                title: 'Overview',
                icon: OverviewIcon,
                link: routes.slo,
                iconSize: 16,
                itemWrapper: menuItemWrapper,
                current: pathname === routes.home || pathname === routes.slo || pathname === routes.zoneDetails,
            },
            {
                id: 'incidents',
                title: 'Incidents',
                icon: IncidentIcon,
                link: routes.incidents,
                iconSize: 16,
                itemWrapper: menuItemWrapper,
                current: pathname === routes.incidents,
            },
            {
                id: 'missing-nodes',
                title: 'Missing Nodes',
                icon: MissingNodesIcon,
                link: routes.missingNodes,
                iconSize: 16,
                itemWrapper: menuItemWrapper,
                current: pathname === routes.missingNodes,
            },
        ] as AsideHeaderProps['menuItems'];
    }, [pathname]);

    return (
        <AsideHeader
            installationType={'internal'}
            onLogoIconClick={toHome}
            logoText="Tentacles"
            logoIconSize={30}
            logoHref="/"
            logoIcon={TentaclesIcon}
            menuItems={menuItems}
            renderContent={() => props.children}
            renderFooter={({isCompact, asideRef}) => <>{<MenuUser isCompact={isCompact} popupAnchor={asideRef} />}</>}
        />
    );
};
