import React from 'react';
import {Link} from 'react-router-dom';
import {AllocationZoneStore} from '../../stores/AllocationZoneStore';
import {observer} from 'mobx-react';
import routes from '../../routes';
import {Panel} from '../../components/Panel/Panel';

import styles from './AllocationZone.module.css';
import sessionsStyles from './Sessions.module.css';
import {Loader, Tabs} from '@yandex-data-ui/common';
import {Monitoring} from './Monitoring/Monitoring';
import {Overview} from './Overview/Overview';
import {Checks} from './Checks/Checks';
import {TABS, TITLE} from './constants';
import {PipelineView} from './Pipeline/PipelineView';
import {useStorePolling} from '../../hooks/useStorePolling';
import {useSearchParam} from '../../hooks/useSearchParam';
import {useDoctitle} from '../../hooks/useDoctitle';
import {ExternalLink} from '../../components/ExternalLink/ExternalLink';
import colors from "../../design/colors.module.css";

export const AllocationZone = observer(() => {
    const [zoneName] = useSearchParam('name');
    const [activeTab, setActiveTab] = useSearchParam('tab');

    if (!zoneName) {
        throw new Error('Zone name must be in search parameter "name"');
    }

    const [store, setStore] = React.useState(() => new AllocationZoneStore(zoneName));
    const setActiveTabCallback = React.useCallback(
        (value: string) => {
            const tab = value === 'overview' ? undefined : value;
            setActiveTab(tab);
        },
        [setActiveTab]
    );

    React.useEffect(() => {
        if (zoneName === store.id) {
            return;
        }

        setStore(new AllocationZoneStore(zoneName));
    }, [zoneName, store, setStore]);

    useStorePolling(store);
    useDoctitle(`${TITLE[zoneName]} ${activeTab || ''}`.trim());

    return (
        <div style={{position: 'relative', minHeight: '100%'}}>
            <Panel style={{}}>
                <div style={{color: 'var(--yc-color-text-hint)'}}>
                    <Link className={'link'} style={{color: 'var(--yc-color-text-hint)'}} to={routes.slo}>
                        slo
                    </Link>{' '}
                    {'>'} {zoneName}
                </div>
                <div>
                    <h2 style={{marginBottom: '8px'}}>{TITLE[zoneName]}</h2>
                    <div style={{display: 'flex'}}>
                        <span style={{paddingRight: '4px', color: colors.textSecondary}}>Useful links:</span>
                        <ExternalLink
                            className={styles.LinksItem}
                            href={`https://juggler.yandex-team.ru/check_details/?host=${zoneName}&service=timestamp-age`}
                        >
                            Juggler check
                        </ExternalLink>

                        <ExternalLink
                            className={styles.LinksItem}
                            href={`https://nanny.yandex-team.ru/ui/#/services/catalog/${zoneName}/`}
                        >
                            Nanny service
                        </ExternalLink>

                        <ExternalLink
                            className={styles.LinksItem}
                            href={`https://wiki.yandex-team.ru/runtime-cloud/rtc-sla-tentacles/#agentskiekontejjneryvnanny`}
                        >
                            Wiki
                        </ExternalLink>
                    </div>
                </div>
            </Panel>
            <Tabs
                className={sessionsStyles.Tabs}
                items={TABS}
                activeTab={activeTab}
                onSelectTab={setActiveTabCallback}
            />
            {activeTab === 'pipeline' ? (
                !store.pipeline || store.slices.size === 0 ? (
                    <Loader />
                ) : (
                    <PipelineView pipeline={store.pipeline} slices={store.slices} zoneId={store.id} />
                )
            ) : activeTab === 'monitoring' ? (
                <Monitoring
                    allocationZoneId={store.id}
                    reallocation={Boolean(store.features?.reallocation.enable)}
                    daemonset={Boolean(store.features?.unusedNodes.enable)}
                />
            ) : activeTab === 'checks' ? (
                <Checks zoneName={store.id} />
            ) : (
                <Overview store={store} />
            )}
        </div>
    );
});
