import React from 'react';
import {Label, Tooltip} from '@yandex-data-ui/common';
import {Link} from 'react-router-dom';

import {NTentaclesApi} from '../../../api/tentacles/stubs/tentacles';
import {Features, Pipeline} from '../../../stores/AllocationZoneStore';
import routes from '../../../routes';
import {HelpIcon} from '../../../components/icons/HelpIcon';
import colors from '../../../design/colors.module.css';
import {ALLOCATOR_HELP, AVAILABILITY_HELP, HQ_HELP, JUGGLER_HELP, REDEPLOYMENT_HELP, WALLE_HELP} from './constants';

interface FeaturesViewProps {
    allocationZone: string;
    pipeline: Pipeline;
    features: Features;
}

function isOk(status?: NTentaclesApi.EFeatureStatus) {
    return status === NTentaclesApi.EFeatureStatus.OK;
}

function toPercent(value: number) {
    return Math.floor(10000 * value) / 100;
}

function filteredPodsLink(zoneName: string, filter: NTentaclesApi.EPodFilter) {
    return `${routes.zoneDetails}?name=${zoneName}&tab=pipeline&podsFilter=${NTentaclesApi.EPodFilter[filter]}`;
}

export const FeaturesView: React.FunctionComponent<FeaturesViewProps> = (props) => {
    const {pipeline, features, allocationZone} = props;

    return (
        <div>
            <div style={{padding: '10px'}}>
                <div style={{display: 'flex', alignItems: 'center'}}>
                    <div style={{flexGrow: 1, paddingBottom: '4px'}}>
                        <div>
                            Walle{' '}
                            <Tooltip htmlContent={WALLE_HELP}>
                                <HelpIcon color={colors.textHint} />
                            </Tooltip>
                        </div>
                    </div>
                    {() => {
                        const exludedTotal =
                            pipeline.walle.excluded_probation_or_free_nodes_count +
                            pipeline.walle.excluded_decommissioned_nodes_count;
                        return <Label theme={exludedTotal === 0 ? 'success' : 'danger'}>{exludedTotal} exluded</Label>;
                    }}
                </div>

                <div>
                    {pipeline.walle.excluded_decommissioned_nodes_count !== 0 && (
                        <div>
                            <Link
                                className={'link link_theme_normal'}
                                to={filteredPodsLink(
                                    allocationZone,
                                    NTentaclesApi.EPodFilter.EXCLUDED_DECOMMISSIONED_NODES
                                )}
                            >
                                {pipeline.walle.excluded_decommissioned_nodes_count} nodes is decommissioned
                            </Link>
                        </div>
                    )}
                    {pipeline.walle.excluded_probation_or_free_nodes_count !== 0 && (
                        <div>
                            <Link
                                className={'link link_theme_normal'}
                                to={filteredPodsLink(
                                    allocationZone,
                                    NTentaclesApi.EPodFilter.EXCLUDED_PROBATION_OR_FREE_NODES
                                )}
                            >
                                {pipeline.walle.excluded_probation_or_free_nodes_count} nodes is free or on probation
                            </Link>
                        </div>
                    )}
                    {pipeline.walle.excluded_probation_or_free_nodes_count +
                        pipeline.walle.excluded_decommissioned_nodes_count ===
                        0 && <div style={{color: 'var(--yc-color-text-secondary)'}}>Everything is fine</div>}
                </div>
            </div>
            <div style={{padding: '10px', borderTop: '1px solid var(--yc-color-line-generic)'}}>
                <div style={{display: 'flex', alignItems: 'center'}}>
                    <div style={{flexGrow: 1, paddingBottom: '4px'}}>
                        <div>
                            Allocator{' '}
                            <Tooltip htmlContent={ALLOCATOR_HELP}>
                                <HelpIcon color={colors.textHint} />
                            </Tooltip>
                        </div>
                        <div style={{fontSize: '10px', color: 'var(--yc-color-text-secondary)'}}>
                            Total: {pipeline.allocator.total_pods_count}
                        </div>
                    </div>
                </div>
                <div>
                    {pipeline.allocator.pods_scheduling_errors !== 0 && (
                        <div>
                            <Link
                                className={'link link_theme_normal'}
                                to={filteredPodsLink(allocationZone, NTentaclesApi.EPodFilter.PODS_SCHEDULING_ERRORS)}
                            >
                                {pipeline.allocator.pods_scheduling_errors} pods scheduling errors
                            </Link>
                        </div>
                    )}
                    {pipeline.allocator.pods_scheduling_errors === 0 && (
                        <div style={{color: 'var(--yc-color-text-secondary)'}}>Everything is fine</div>
                    )}
                </div>
            </div>
            <div style={{padding: '10px', borderTop: '1px solid var(--yc-color-line-generic)'}}>
                <div style={{display: 'flex', alignItems: 'center'}}>
                    <div style={{flexGrow: 1, paddingBottom: '4px'}}>
                        <div>
                            HQ{' '}
                            <Tooltip htmlContent={HQ_HELP}>
                                <HelpIcon color={colors.textHint} />
                            </Tooltip>
                        </div>
                        <div style={{fontSize: '10px', color: 'var(--yc-color-text-secondary)'}}>
                            Total: {pipeline.hq.total_pods}
                        </div>
                    </div>
                </div>
            </div>
            <div style={{padding: '10px', borderTop: '1px solid var(--yc-color-line-generic)'}}>
                <div style={{display: 'flex', alignItems: 'center'}}>
                    <div style={{flexGrow: 1, paddingBottom: '4px'}}>
                        <div>
                            Juggler{' '}
                            <Tooltip htmlContent={JUGGLER_HELP}>
                                <HelpIcon color={colors.textHint} />
                            </Tooltip>
                        </div>
                        <div style={{fontSize: '10px', color: 'var(--yc-color-text-secondary)'}}>
                            Total: {pipeline.juggler.total_pods}
                        </div>
                    </div>
                </div>
                <div>
                    {pipeline.juggler.excluded_not_visible !== 0 && (
                        <div>
                            <Link
                                className={'link link_theme_normal'}
                                to={filteredPodsLink(allocationZone, NTentaclesApi.EPodFilter.EXCLUDED_NOT_VISIBLE)}
                            >
                                {pipeline.juggler.excluded_not_visible} nodes is not visible
                            </Link>
                        </div>
                    )}
                    {pipeline.juggler.excluded_not_present !== 0 && (
                        <div>
                            <Link
                                className={'link link_theme_normal'}
                                to={filteredPodsLink(allocationZone, NTentaclesApi.EPodFilter.EXCLUDED_NOT_PRESENT)}
                            >
                                {pipeline.juggler.excluded_not_present} nodes is not present
                            </Link>
                        </div>
                    )}
                </div>
            </div>
            <div style={{padding: '10px', borderTop: '1px solid var(--yc-color-line-generic)'}}>
                <div style={{display: 'flex', alignItems: 'center'}}>
                    <div style={{flexGrow: 1, paddingBottom: '4px'}}>
                        <div>
                            Availability{' '}
                            <Tooltip htmlContent={AVAILABILITY_HELP}>
                                <HelpIcon color={colors.textHint} />
                            </Tooltip>
                        </div>
                        <div style={{fontSize: '10px', color: 'var(--yc-color-text-secondary)'}}>
                            Total: {features.availability.total_count}, Ok: {features.availability.ok_count}
                        </div>
                    </div>
                    <Label theme={isOk(features.availability.status) ? 'success' : 'danger'}>
                        {toPercent(features.availability.ok_count / features.availability.total_count)}%
                    </Label>
                </div>
                <div>
                    {pipeline.availability.excluded_not_reachable !== 0 && (
                        <div>
                            <Link
                                className={'link link_theme_normal'}
                                to={filteredPodsLink(allocationZone, NTentaclesApi.EPodFilter.EXCLUDED_NOT_REACHABLE)}
                            >
                                {pipeline.availability.excluded_not_reachable} nodes is not reachable
                            </Link>
                        </div>
                    )}
                    {pipeline.availability.excluded_not_reachable === 0 && (
                        <div style={{color: 'var(--yc-color-text-secondary)'}}>Everything is fine</div>
                    )}
                </div>
            </div>
            <div style={{padding: '10px', borderTop: '1px solid var(--yc-color-line-generic)'}}>
                <div style={{display: 'flex', alignItems: 'center'}}>
                    <div style={{flexGrow: 1, paddingBottom: '4px'}}>
                        <div>
                            Redeployment{' '}
                            <Tooltip htmlContent={REDEPLOYMENT_HELP}>
                                <HelpIcon color={colors.textHint} />
                            </Tooltip>
                        </div>
                        <div style={{fontSize: '10px', color: 'var(--yc-color-text-secondary)'}}>
                            Total: {pipeline.redeployment.total_count}, Ok: {pipeline.redeployment.ok_count}
                        </div>
                    </div>
                    <Label theme={isOk(features.redeploy.status) ? 'success' : 'danger'}>
                        {toPercent(pipeline.redeployment.ok_count / pipeline.redeployment.total_count)}%
                    </Label>
                </div>
            </div>
        </div>
    );
};
