import Xrange from 'highcharts/modules/xrange';
import Highcharts from 'highcharts';
import {Session} from '../../../stores/AllocationZoneStore';
import React from 'react';

Xrange(Highcharts);

const timezone = new Date().getTimezoneOffset();

interface SessionViewProps {
    sessions: Session[];
}

const SESSIONS_ORDER = new Map([
    ['redeployment', 0],
    ['reallocation', 1],
]);

const SESSIONS_COLOR = new Map([
    ['redeployment', 'var(--yc-color-infographics-positive-heavy)'],
    ['reallocation', 'var(--yc-color-infographics-info-heavy)'],
]);

export const SessionsChart: React.FunctionComponent<SessionViewProps> = (props) => {
    const ref = React.useRef<HTMLDivElement>(null);
    const [chart, setChart] = React.useState<Highcharts.Chart | undefined>();

    React.useEffect(() => {
        if (!ref.current) {
            return;
        }

        const initChart = Highcharts.chart(ref.current as HTMLElement, {
            time: {
                timezoneOffset: timezone,
            },
            chart: {
                type: 'x-range',
            },
            title: {
                text: '',
            },
            tooltip: {
                enabled: false,
            },
            xAxis: {
                type: 'datetime',
                labels: {
                    style: {
                        fontSize: '8px',
                    },
                },
                tickLength: 6,
                plotLines: [
                    {
                        id: 'now',
                        color: '#FF0000',
                        width: 1,
                        value: Date.now(),
                    },
                ],
            },
            yAxis: {
                title: {
                    text: '',
                },
                gridLineWidth: 0,
                categories: ['redeployment', 'reallocation'],

                labels: {
                    enabled: false,
                    style: {
                        fontSize: '10px',
                    },
                },
                reversed: false,
            },
            series: [
                {
                    animation: false,
                    type: 'xrange',
                    name: 'Session',
                    showInLegend: false,
                },
            ],
            credits: {
                enabled: false,
            },
        });

        setChart(initChart);

        return () => {
            setChart(undefined);
            initChart.destroy();
        };
    }, [ref]);

    React.useEffect(() => {
        if (!chart || !chart.series || !chart.series[0]) {
            return;
        }

        const data = props.sessions.map((item) => {
            return {
                x: item.session.start_ts * 1000,
                x2: item.session.end_ts * 1000,
                y: SESSIONS_ORDER.get(item.type),
                color: SESSIONS_COLOR.get(item.type),
            };
        });
        chart.series[0].setData(data);
        chart.xAxis[0].removePlotLine('now');
        chart.xAxis[0].addPlotLine({
            id: 'now',
            color: '#FF0000',
            width: 1,
            value: Date.now(),
        });
        chart.xAxis[0].update({softMax: Date.now() + 60000});
    }, [chart, props.sessions]);

    return <div ref={ref} style={{height: '100px'}} />;
};
